<?php

declare(strict_types=1);

namespace Drupal\coveo\Plugin\Coveo\SecurityProvider;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Session\AccountInterface;
use Drupal\coveo\Entity\CoveoSearchComponent;
use Drupal\coveo\Event\CoveoTokenAlter;
use Drupal\coveo\Plugin\CoveoIdentityProviderPluginInterface;
use Drupal\coveo\Plugin\CoveoSecurityProviderPluginInterface;
use NecLimDul\Coveo\SearchApi\Model\RestTokenParams;
use NecLimDul\Coveo\SearchApi\Model\RestUserId;
use Neclimdul\OpenapiPhp\Helper\Logging\Error;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Base class for an abstract secured user security provider.
 *
 * Abstracts generating the API arguments so a provider doesn't have to contain
 * the undocumented logic of the model definitions.
 */
abstract class AbstractSecuredUserProvider extends PluginBase implements CoveoSecurityProviderPluginInterface, CoveoIdentityProviderPluginInterface, ContainerFactoryPluginInterface {

  use LoggerChannelTrait;

  /**
   * Constructs a EmailProvider security provider plugin.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    LoggerChannelFactoryInterface $loggerChannelFactory,
    private readonly EventDispatcherInterface $eventDispatcher,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->setLoggerFactory($loggerChannelFactory);
  }

  /**
   * {@inheritDoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory'),
      $container->get('event_dispatcher'),
    );
  }

  /**
   * {@inheritDoc}
   */
  #[\Override]
  public function generateToken(CoveoSearchComponent $search, AccountInterface $account): string {

    $parameters = $this->generateParameters($account);

    // Trigger event so additional analytics or permissions can be added.
    $this->eventDispatcher->dispatch(new CoveoTokenAlter(
      $parameters,
      $account,
      $search,
    ));

    $response = $search->getSearchApi()->token(
      $parameters,
      $search->getOrganizationId(),
    );
    if ($response->isSuccess()) {
      $token = $response->getData()->getToken();
    }
    else {
      Error::logError($this->getLogger('coveo'), $response);
      $token = 'fail';
    }

    return $token;
  }

  /**
   * Generate token request parameters.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   Account for generating a token.
   *
   * @return \NecLimDul\Coveo\SearchApi\Model\RestTokenParams
   *   Token generation parameters.
   */
  public function generateParameters(AccountInterface $account): RestTokenParams {
    $params = new RestTokenParams();
    $params->setUserIds([
      new RestUserId([
        'name' => $this->getName($account),
        'provider' => $this->getIdentityProviderId(),
        'type' => 'User',
        'infos' => [],
        'auth_cookie' => '',
        'password' => '',
      ]),
    ]);

    return $params;
  }

  /**
   * The "name" of the user.
   *
   * In Coveo authentication provide parlance this is actually the ID, either
   * an email or other ID.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   Account for generating a token.
   *
   * @return string
   *   The account name/id.
   */
  abstract protected function getName(AccountInterface $account): string;

}
