<?php

declare(strict_types=1);

namespace Drupal\coveo_atomic\Twig;

use Drupal\coveo\Entity\CoveoSearchComponent;
use Twig\Extension\AbstractExtension;
use Twig\TwigFilter;

/**
 * Provide twig extensions to help with Coveo Atomic development.
 */
class AtomicExtension extends AbstractExtension {

  /**
   * Generates a list of all Twig filters that this extension defines.
   *
   * @return array
   *   A key/value array that defines custom Twig filters.
   */
  public function getFilters(): array {
    return [
      'atomic_fields' => new TwigFilter(
        'atomic_fields',
        [self::class, 'atomicFieldsFilter'],
        ['is_variadic' => TRUE],
      ),
      'atomic_field' => new TwigFilter(
        'atomic_field',
        [self::class, 'atomicFieldFilter'],
        ['is_variadic' => TRUE],
      ),
    ];
  }

  /**
   * Convert a field list to Coveo field list. Applies magic prefixing logic.
   *
   * Note: If you don't need any magic prefixing, the json_encode filter should
   * be enough.
   *
   * @param string[] $field_list
   *   Field list from input.
   * @param array{0?: string} $args
   *   List of arguments.
   *
   * @return string
   *   Rendered field list for Coveo.
   */
  public static function atomicFieldsFilter(array $field_list, array $args = []): string {
    if (empty($args)) {
      // Error.
      return '[]';
    }
    // Extract arguments.
    [$search] = $args;
    $search_components = CoveoSearchComponent::load($search);
    if ($search_components === NULL) {
      // Error.
      return '[]';
    }

    return json_encode(iterator_to_array(
      $search_components
        ->getOrganization()
        ->getFieldConverter()
        ->convertFieldNames($field_list)
    ));
  }

  /**
   * Convert a field to Coveo field name. Applies magic prefixing logic.
   *
   * @param string $field_name
   *   Field name from input.
   * @param array{0?: string} $args
   *   List of arguments.
   *
   * @return string
   *   Rendered field for Coveo.
   */
  public static function atomicFieldFilter(string $field_name, array $args = []): string {
    if (empty($args)) {
      // Error.
      return '';
    }
    // Extract arguments.
    [$search] = $args;
    $search_components = CoveoSearchComponent::load($search);
    if ($search_components === NULL) {
      // Error.
      return '';
    }

    return $search_components
      ->getOrganization()
      ->getFieldConverter()
      ->convertFieldName($field_name);
  }

}
