<?php

declare(strict_types=1);

namespace Drupal\coveo_search_api;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\coveo_search_api\Plugin\search_api\backend\SearchApiCoveoBackend;
use Drupal\search_api\ServerInterface;

/**
 * Cover servers storage helper.
 */
class CoveoServers {

  /**
   * Entity storage for SearchAPI servers.
   */
  private EntityStorageInterface $serverStorage;

  public function __construct(EntityTypeManagerInterface $typeManager) {
    $this->serverStorage = $typeManager->getStorage('search_api_server');
  }

  /**
   * Get a list of Coveo SearchAPI servers.
   *
   * @param string $organization_name
   *   Drupal organization id.
   *
   * @return array<string,\Drupal\search_api\Entity\Server>
   *   A list of Coveo servers keyed by their id.
   */
  public function getSearchApiServers(string $organization_name = ''): array {
    /** @var array<string,\Drupal\search_api\Entity\Server> $servers */
    $servers = $this->serverStorage->loadByProperties([
      'backend' => 'coveo',
    ]);
    if ($organization_name) {
      $servers = array_filter(
        $servers,
        // Filter backends to this specific org.
        fn(ServerInterface $server) =>
          $server->getBackendConfig()['organization_name'] === $organization_name,
      );
    }
    return $servers;
  }

  /**
   * Get a list of Coveo SearchAPI backends.
   *
   * @param string $organization_name
   *   Drupal organization id.
   *
   * @return array<string,\Drupal\coveo_search_api\Plugin\search_api\backend\SearchApiCoveoBackend>
   *   A list of Coveo backends keyed by their id.
   */
  public function getSearchBackends(string $organization_name = ''): array {
    return array_map(
      function (ServerInterface $server): SearchApiCoveoBackend {
        $backend = $server->getBackend();
        assert($backend instanceof SearchApiCoveoBackend);
        return $backend;
      },
      $this->getSearchApiServers($organization_name),
    );
  }

  /**
   * Get a list of Coveo SearchAPI servers.
   *
   * @param string $organization_name
   *   Drupal organization id.
   *
   * @return array<\Drupal\search_api\IndexInterface>
   *   A list of Coveo indexes keyed by their id.
   */
  public function getSearchIndexes(string $organization_name = ''): array {
    $indexes = [];
    foreach ($this->getSearchApiServers($organization_name) as $server) {
      foreach ($server->getIndexes() as $index) {
        $indexes[] = $index;
      }
    }
    return $indexes;
  }

}
