<?php

declare(strict_types=1);

namespace Drupal\coveo_search_api;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\coveo\Event\CoveoOrganizationSync;
use Drupal\coveo_secured_search\Event\CoveoSecurityProviderAlter;
use Drupal\coveo_secured_search\Plugin\search_api\backend\CoveoIdentityBackend;
use Drupal\search_api\ServerInterface;
use NecLimDul\Coveo\SecurityCache\Model\SecurityProviderReferenceModel;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Coveo subscriber.
 */
readonly class CoveoSubscriber implements EventSubscriberInterface {

  public function __construct(
    private SyncFields $syncFields,
    private EntityTypeManagerInterface $typeManager,
  ) {
  }

  /**
   * {@inheritDoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      CoveoOrganizationSync::class => 'syncFields',
      CoveoSecurityProviderAlter::class => 'addProviderReferences',
    ];
  }

  /**
   * Sync fields with Coveo using SearchAPI push sources.
   *
   * @param \Drupal\coveo\Event\CoveoOrganizationSync $event
   *   Coveo organization sync event.
   */
  public function syncFields(CoveoOrganizationSync $event): void {
    $this->syncFields->syncOrganization($event->organization);
  }

  /**
   * Sync fields with Coveo using SearchAPI push sources.
   *
   * @param \Drupal\coveo_secured_search\Event\CoveoSecurityProviderAlter $event
   *   Coveo provider alter event.
   */
  public function addProviderReferences(CoveoSecurityProviderAlter $event): void {
    // Populate references.
    $references = array_filter(
      $this->getIdentityBackends(),
      fn(CoveoIdentityBackend $backend) => $backend->getProviderPluginId() === $event->securityProvider->id(),
    );
    // @todo how do we handle this?
    foreach ($references as $backend) {
      $references[] = new SecurityProviderReferenceModel([
        'id' => $backend->getOrganization()->getPushSourceId(),
        'type' => SecurityProviderReferenceModel::TYPE_SOURCE,
      ]);
    }
    $event->securityProviderModel->setReferencedBy($references);
  }

  /**
   * Get Coveo identity backends.
   *
   * @return \Drupal\coveo_secured_search\Plugin\search_api\backend\CoveoIdentityBackend[]
   *   Coveo identity backends.
   */
  private function getIdentityBackends(): array {
    $server_storage = $this->typeManager->getStorage('search_api_server');

    $servers = $server_storage->loadByProperties([
      'backend' => 'coveo_identity',
    ]);
    return array_map(
      function (ServerInterface $server): CoveoIdentityBackend {
        $backend = $server->getBackend();
        assert($backend instanceof CoveoIdentityBackend);
        return $backend;
      },
      $servers
    );
  }

}
