<?php

declare(strict_types=1);

namespace Drupal\coveo_secured_search\Entity;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\coveo\Entity\CoveoOrganizationInterface;
use Drupal\coveo\Entity\CoveoSearchComponent;
use Drupal\coveo_secured_search\Plugin\CoveoCustomSecurityProviderPluginInterface;

/**
 * Coveo Security Provider entity.
 */
interface CoveoCustomSecurityProviderInterface extends ConfigEntityInterface {

  /**
   * Description of the security provider.
   *
   * @return string|null
   *   Description of the security provider.
   */
  public function getDescription(): string|null;

  /**
   * The security provider plugin ID.
   *
   * @return string|null
   *   The security provider plugin ID.
   */
  public function getSecurityProviderPluginId(): string|null;

  /**
   * The security provider ID in Coveo.
   *
   * This is the machine name that identities the Coveo provider.
   *
   * @return string|null
   *   The associated security provider ID in Coveo.
   */
  public function getSecurityProviderId(): string|null;

  /**
   * Get the connected Coveo security plugin.
   *
   * @return \Drupal\coveo_secured_search\Plugin\CoveoCustomSecurityProviderPluginInterface
   *   Connected Coveo Security provider plugin.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   *   Associated security provider doesn't exist.
   */
  public function getSecurityProvider(): CoveoCustomSecurityProviderPluginInterface;

  /**
   * Get the list of push source IDs.
   *
   * @return string[]
   *   The list of push sources Search API IDs.
   */
  public function getPushSourceIds(): array;

  /**
   * Get the internal organization machine name.
   *
   * @return string|null
   *   Organization config machine name.
   */
  public function getOrganizationName(): string|null;

  /**
   * Get the Coveo machine name.
   *
   * This is the machine name used by Coveo to identify this source. If not
   * provided Drupal will create one based on the Drupal machine name.
   *
   * @return string|null
   *   The Coveo machine name.
   */
  public function getCoveoName(): string|null;

  /**
   * Get the associated organization config.
   *
   * @return \Drupal\coveo\Entity\CoveoOrganizationInterface|null
   *   Organization config.
   */
  public function getOrganization(): CoveoOrganizationInterface|null;

  /**
   * Generate a token for a user.
   *
   * @param \Drupal\coveo\Entity\CoveoSearchComponent $search
   *   Related Coveo search definition.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user to generate a token for. Generally the current user.
   *
   * @return string
   *   Search token.
   */
  public function generateToken(CoveoSearchComponent $search, AccountInterface $account): string;

}
