<?php

declare(strict_types=1);

namespace Drupal\coveo_secured_search\Plugin\Coveo\SecurityProvider;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\coveo\Attribute\CoveoSecurityProvider;
use Drupal\coveo\Entity\CoveoSearchComponent;
use Drupal\coveo\Plugin\CoveoSecurityProviderPluginInterface;
use Drupal\coveo_secured_search\Entity\CoveoCustomSecurityProviderInterface;
use Drupal\coveo_secured_search\Plugin\CoveoCustomSecurityProviderPluginInterface;
use Drupal\coveo_secured_search\Plugin\Derivative\CustomSecurityProviderDeriver;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Email Provider security provider.
 *
 * Generates tokens for the out-of-the-box Coveo Email Token Security Provider.
 */
#[CoveoSecurityProvider(
  id: 'custom_provider',
  title: new TranslatableMarkup('Custom Providers'),
  description: new TranslatableMarkup("..."),
  category: 'custom',
  deriver: CustomSecurityProviderDeriver::class,
)]
class CustomProvider extends PluginBase implements CoveoSecurityProviderPluginInterface, ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly CoveoCustomSecurityProviderInterface $coveoCustomSecurityProvider,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritDoc}
   */
  #[\Override]
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    [, $id] = explode(':', $plugin_id, 2);
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
        ->getStorage('coveo_custom_security_provider')
        ->load($id),
    );
  }

  /**
   * {@inheritDoc}
   */
  #[\Override]
  public function generateToken(
    CoveoSearchComponent $search,
    AccountInterface $account,
  ): string {
    return $this->coveoCustomSecurityProvider->generateToken($search, $account);
  }

  /**
   * {@inheritDoc}
   */
  #[\Override]
  public function getIdentityProviderId(): string {
    return $this->coveoCustomSecurityProvider->getSecurityProviderId();
  }

  /**
   * Get the connected custom security provider plugin.
   */
  public function getCustomSecurityProviderPlugin(): CoveoCustomSecurityProviderPluginInterface {
    return $this->coveoCustomSecurityProvider->getSecurityProvider();
  }

}
