<?php

declare(strict_types=1);

namespace Drupal\coveo\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\coveo\API\OrganizationApiHelper;
use Drupal\coveo\Coveo\Index;
use Drupal\coveo\Coveo\IndexHelperFactory;
use Drupal\coveo\Event\CoveoOrganizationSync;
use Drupal\coveo\FieldConverter;
use NecLimDul\Coveo\SecurityCache\Api\SecurityProvidersApi;

/**
 * Defines Coveo Organization configuration entity.
 *
 * @ConfigEntityType(
 *   id = "coveo_organization",
 *   label = @Translation("Coveo Organization"),
 *   label_collection = @Translation("Coveo Organizations"),
 *   label_singular = @Translation("Coveo organization"),
 *   label_plural = @Translation("Coveo organizations"),
 *   label_count = @PluralTranslation(
 *     singular = "@count Coveo organization",
 *     plural = "@count Coveo organizations",
 *   ),
 *   handlers = {
 *     "form" = {
 *       "add" = "Drupal\coveo\Form\Organization\CoveoOrganizationForm",
 *       "edit" = "Drupal\coveo\Form\Organization\CoveoOrganizationForm",
 *       "delete" = "Drupal\coveo\Form\Organization\CoveoOrganizationDeleteForm",
 *     },
 *     "list_builder" = "Drupal\coveo\OrganizationListBuilder",
 *     "storage" = "Drupal\coveo\OrganizationStorage",
 *   },
 *   admin_permission = "administer coveo search",
 *   config_prefix = "organization",
 *   entity_keys = {
 *     "id" = "name",
 *     "label" = "label",
 *   },
 *   links = {
 *     "view" = "/admin/config/search/coveo/organization/manage/{coveo_organization}",
 *     "edit-form" = "/admin/config/search/coveo/organization/manage/{coveo_organization}/edit",
 *     "delete-form" = "/admin/config/search/coveo/organization/manage/{coveo_organization}/delete",
 *     "collection" = "/admin/config/search/coveo/organization",
 *   },
 *   config_export = {
 *     "name",
 *     "label",
 *     "organization_id" = "organization_id",
 *     "read_only" = "read_only",
 *     "auto_sync" = "auto_sync",
 *     "prefix" = "prefix",
 *     "push_key" = "push_key",
 *     "push_source_id" = "push_source_id",
 *   }
 * )
 */
class CoveoOrganization extends ConfigEntityBase implements CoveoOrganizationInterface {

  /**
   * The local machine name of the Coveo organization.
   */
  protected string $name;

  /**
   * The Coveo organization label.
   */
  protected string $label;

  /**
   * Coveo organization ID.
   */
  protected ?string $organization_id;

  /**
   * Coveo read only toggle.
   */
  protected bool $read_only = FALSE;

  /**
   * Should auto-sync things like fields.
   */
  protected bool $auto_sync = FALSE;

  /**
   * Sub-environment prefix.
   */
  protected ?string $prefix = '';

  /**
   * Coveo push key.
   */
  protected ?string $push_key;

  /**
   * Coveo push source id.
   */
  protected ?string $push_source_id;

  /**
   * Coveo security provider API connection.
   */
  private SecurityProvidersApi $securityProviderApi;

  /**
   * API service injection helper.
   */
  private OrganizationApiHelper $apiHelper;

  /**
   * Index helper factory.
   */
  private mixed $getIndexHelperFactory;

  /**
   * {@inheritdoc}
   */
  public function id(): string|null {
    return $this->name ?? NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function getOrganizationId(): string|null {
    return $this->organization_id ?? NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function isReadOnly(): bool {
    return $this->read_only;
  }

  /**
   * {@inheritDoc}
   */
  public function autoSync(): bool {
    return $this->auto_sync;
  }

  /**
   * {@inheritDoc}
   */
  public function getPrefix(): string {
    return $this->prefix ?? '';
  }

  /**
   * {@inheritDoc}
   */
  public function getPushKey(): string|null {
    return $this->push_key ?? NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function getPushSourceId(): string|null {
    return $this->push_source_id ?? NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function sync(): void {
    // Trigger event so additional sync actions can take place.
    \Drupal::service('event_dispatcher')->dispatch(new CoveoOrganizationSync(
      $this,
    ));
  }

  /**
   * {@inheritDoc}
   */
  public function getSecurityProviderApi(): SecurityProvidersApi {
    if (!isset($this->securityProviderApi)) {
      $this->securityProviderApi = $this->securityCacheCreate(SecurityProvidersApi::class);
    }
    return $this->securityProviderApi;
  }

  /**
   * {@inheritDoc}
   */
  public function fieldApiCreate(string $class): mixed {
    return $this->getApiHelper()->fieldApiFactory
      ->create(
        $class,
        $this->getPushKey(),
      );
  }

  /**
   * {@inheritDoc}
   */
  public function pushApiCreate(string $class): mixed {
    return $this->getApiHelper()->pushApiFactory
      ->create(
        $class,
        $this->getPushKey(),
      );
  }

  /**
   * {@inheritDoc}
   */
  public function securityCacheCreate(string $class): mixed {
    return $this->getApiHelper()->securityCacheFactory
      ->create(
        $class,
        $this->getPushKey(),
      );
  }

  /**
   * Get an Organization API helper.
   *
   * @return \Drupal\coveo\API\OrganizationApiHelper
   *   Ano Organization API Helper.
   */
  private function getApiHelper(): OrganizationApiHelper {
    if (!isset($this->apiHelper)) {
      $this->apiHelper = \Drupal::service('coveo.org_api_helper');
    }
    return $this->apiHelper;
  }

  /**
   * {@inheritDoc}
   */
  public function getFieldConverter(): FieldConverter {
    return new FieldConverter($this->getPrefix());
  }

  /**
   * Get an Organization API helper.
   *
   * @return \Drupal\coveo\Coveo\IndexHelperFactory
   *   Ano Organization API Helper.
   */
  private function getIndexHelperFactory(): IndexHelperFactory {
    if (!isset($this->getIndexHelperFactory)) {
      $this->getIndexHelperFactory = \Drupal::service('coveo.index_helper_factory');
    }
    return $this->getIndexHelperFactory;
  }

  /**
   * {@inheritDoc}
   */
  public function getIndexHelper(): Index {
    return $this->getIndexHelperFactory()->createIndex($this);
  }

}
