<?php

declare(strict_types=1);

namespace Drupal\coveo\Entity;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Session\AccountInterface;
use Drupal\coveo\Plugin\CoveoSecurityProviderPluginInterface;
use NecLimDul\Coveo\SearchApi\Api\SearchV2Api;
use Symfony\Component\HttpFoundation\Request;

/**
 * Defines Coveo Search configuration entity.
 *
 * Defines configuration for exposing searches through user key generation.
 *
 * @ConfigEntityType(
 *   id = "coveo_search_component",
 *   label = @Translation("Coveo Search Component"),
 *   label_collection = @Translation("Coveo Search Components"),
 *   label_singular = @Translation("Coveo search component"),
 *   label_plural = @Translation("Coveo search components"),
 *   label_count = @PluralTranslation(
 *     singular = "@count Coveo search component",
 *     plural = "@count Coveo search components",
 *   ),
 *   handlers = {
 *     "form" = {
 *       "add" = "Drupal\coveo\Form\SearchComponents\CoveoSearchComponentForm",
 *       "edit" = "Drupal\coveo\Form\SearchComponents\CoveoSearchComponentForm",
 *       "delete" = "Drupal\coveo\Form\SearchComponents\CoveoSearchComponentDeleteForm",
 *     },
 *     "list_builder" = "Drupal\coveo\SearchComponentListBuilder",
 *     "storage" = "Drupal\coveo\SearchComponentStorage",
 *   },
 *   admin_permission = "administer coveo search",
 *   config_prefix = "search_component",
 *   entity_keys = {
 *     "id" = "name",
 *     "label" = "label",
 *     "search_key" = "search_key",
 *     "security_provider" = "security_provider",
 *   },
 *   links = {
 *     "edit-form" = "/admin/config/search/coveo/search_components/manage/{coveo_search_component}",
 *     "delete-form" = "/admin/config/search/coveo/search_components/manage/{coveo_search_component}/delete",
 *     "collection" = "/admin/config/search/coveo/search_components",
 *   },
 *   config_export = {
 *     "name",
 *     "label",
 *     "search_key",
 *     "organization_name",
 *     "security_provider",
 *   }
 * )
 */
class CoveoSearchComponent extends ConfigEntityBase implements CoveoSearchComponentInterface {

  /**
   * The name of the Coveo search.
   */
  protected string $name;

  /**
   * The Coveo search label.
   */
  protected string $label;

  /**
   * Internal Coveo organization machine name.
   */
  protected ?string $organization_name;

  /**
   * Search key used for user token generation.
   */
  protected ?string $search_key;

  /**
   * Search API service.
   */
  private SearchV2Api $coveoSearchApi;

  /**
   * {@inheritdoc}
   */
  public function id(): string|null {
    return $this->name ?? NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function searchKey(): string|null {
    return $this->search_key ?? NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function getSearchApi(): SearchV2Api {
    if (!isset($this->coveoSearchApi)) {
      $this->coveoSearchApi = \Drupal::service('coveo.rest.search_api_factory')
        ->create(SearchV2Api::class, $this->searchKey());
    }

    return $this->coveoSearchApi;
  }

  /**
   * {@inheritDoc}
   */
  public function getOrganizationId(): string|null {
    return $this->getOrganization()?->getOrganizationId();
  }

  /**
   * {@inheritDoc}
   */
  public function getOrganizationName(): string|null {
    return $this->organization_name ?? NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function getSecurityProviderId(): string|null {
    return $this->security_provider ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    parent::calculateDependencies();
    $organization = $this->getOrganization();
    $this->addDependency('config', $organization->getConfigDependencyName());
    return $this;
  }

  /**
   * Get the associated organization config.
   *
   * @return \Drupal\coveo\Entity\CoveoOrganizationInterface|null
   *   Organization config.
   */
  public function getOrganization(): CoveoOrganizationInterface|null {
    $id = $this->getOrganizationName();
    if ($id) {
      return $this->entityTypeManager()
        ->getStorage('coveo_organization')
        ->load($id);
    }
    return NULL;
  }

  /**
   * Get the connected Coveo security plugin.
   *
   * @return \Drupal\coveo\Plugin\CoveoSecurityProviderPluginInterface
   *   Connected Coveo Security provider plugin.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   *   Associated service provider doesn't exist.
   */
  public function getSecurityProvider(): CoveoSecurityProviderPluginInterface {
    /** @var \Drupal\coveo\Plugin\CoveoSecurityProviderManager $manager */
    $manager = \Drupal::service('plugin.manager.coveo_security_provider');
    return $manager->createInstance($this->getSecurityProviderId());
  }

  /**
   * Get a search token for a user.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   Related request for storing session data.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user to generate a token for. Generally the current user.
   *
   * @return string|false
   *   Refreshed search token.
   */
  public function getToken(Request $request, AccountInterface $account): false|string {
    $token = FALSE;
    // @todo figure out how to avoid unnecessary token generation
    // Sessions are broken and can't time out for refresh broken tokens so
    // disabled until I can find a fix.
    // $session = $request->getSession();
    // $session->get('coveo_token', FALSE);
    // $session->remove('coveo_token');
    /* @phpstan-ignore-next-line */
    if (!$token) {
      try {
        return $this->getSecurityProvider()
          ->generateToken($this, $account);
      }
      catch (PluginException $e) {
        // @todo log plugin failure.
        // Call through and return false as a failure.
      }
    }
    return $token;
  }

}
