<?php

declare(strict_types=1);

namespace Drupal\coveo;

/**
 * Helper class to manage tasks related to field indexing.
 */
class FieldConverter {

  /**
   * A prefix for all fields to identify as coming from this module.
   */
  const string FIELD_PREFIX = 'drpl';

  /**
   * Item ID field in Coveo results.
   */
  const string COVEO_ITEM_ID_FIELD = self::FIELD_PREFIX . '_search_api_id';

  /**
   * Source ID field in Coveo results.
   *
   * Provided to make it easier to query specific sources.
   */
  const string COVEO_INDEX_ID_FIELD = self::FIELD_PREFIX . '_index_id';

  /**
   * Drupal's prefix value for use in extensions of other automation in Coveo.
   */
  const string COVEO_PREFIX_FIELD = self::FIELD_PREFIX . '_drupal_prefix';

  public function __construct(private readonly string $prefix) {
    // Intentionally blank.
  }

  /**
   * Get the configured prefix for the related fields.
   */
  public function getPrefix(): string {
    // If there isn't an org prefix, use a default to make sure there is
    // something to namespace our fields from built in Coveo fields or other
    // custom fields.
    return $this->prefix ?: self::FIELD_PREFIX;
  }

  /**
   * Check if this field is a Drupal field or another field in Coveo.
   *
   * @param string $field_name
   *   The Coveo field name.
   *
   * @return bool
   *   True if field is owned by Drupal.
   */
  public function isDrupalField(string $field_name): bool {
    return str_starts_with($field_name, $this->getPrefix() . '_');
  }

  /**
   * Convert a Coveo field to its Drupal field name.
   *
   * @param string $coveo_field_name
   *   The Coveo field name.
   *
   * @return string
   *   The Drupal field name.
   */
  public function convertCoveoToDrupal(string $coveo_field_name): string {
    return substr($coveo_field_name, strlen($this->getPrefix() . '_'));
  }

  /**
   * Convert a Drupal field name to a Coveo field name.
   *
   * @param string $drupal_field_name
   *   The Drupal field name.
   *
   * @return string
   *   The Coveo field name.
   */
  public function convertDrupalToCoveo(string $drupal_field_name): string {
    return $this->getPrefix() . '_' . $drupal_field_name;
  }

  /**
   * Check if a field name matches on of our internal system fields.
   *
   * @param string $field_name
   *   Coveo field name.
   *
   * @return bool
   *   True if the field is a system field.
   */
  public function isDrupalSystemField(string $field_name): bool {
    return $field_name === self::COVEO_ITEM_ID_FIELD
      || $field_name === self::COVEO_PREFIX_FIELD
      || $field_name === self::COVEO_INDEX_ID_FIELD;
  }

  /**
   * Is the field a magic Coveo field.
   *
   * @param string $field_name
   *   Index field name.
   *
   * @return bool
   *   True if the field should be magically mapped to internal Coveo field.
   */
  public function isCoveoMagic(string $field_name): bool {
    return str_starts_with($field_name, 'coveo_');
  }

  /**
   * Convert index field name to internal Coveo field.
   *
   * @param string $field_name
   *   The index field name.
   *
   * @return string
   *   Internal Coveo field name.
   */
  public function convertCoveoMagicField(string $field_name): string {
    return substr($field_name, strlen('coveo_'));
  }

  /**
   * Helper to mangle field list with prefixes into actual Coveo names.
   *
   * @param string[] $field_list
   *   A field list.
   *
   * @return \Generator
   *   Generator that can provide converted fields.
   */
  public function convertFieldNames(array $field_list): \Generator {
    foreach ($field_list as $field_name) {
      yield $this->convertFieldName($field_name);
    }
  }

  /**
   * Helper to mangle field list with prefixes into actual Coveo names.
   *
   * @param string $field_name
   *   A field name.
   *
   * @return string
   *   Converted field name.
   */
  public function convertFieldName(string $field_name): string {
    if ($this->isDrupalSystemField($field_name)) {
      return $field_name;
    }
    elseif ($this->isCoveoMagic($field_name)) {
      return $this->convertCoveoMagicField($field_name);
    }
    else {
      return $this->convertDrupalToCoveo($field_name);
    }
  }

}
