<?php

declare(strict_types=1);

namespace Drupal\coveo\Form\Organization;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for adding and editing Coveo Organizations.
 */
class CoveoOrganizationForm extends EntityForm {

  /**
   * The entity being used by this form.
   *
   * @var \Drupal\coveo\Entity\CoveoOrganizationInterface
   */
  protected $entity;

  /**
   * Constructs a base class for Coveo search add and edit forms.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $organizationStorage
   *   The Coveo search entity storage.
   */
  public function __construct(
    protected EntityStorageInterface $organizationStorage,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')->getStorage('coveo_organization'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {

    $form['label'] = [
      '#title' => $this->t('Name'),
      '#type' => 'textfield',
      '#default_value' => $this->entity->label(),
      '#description' => $this->t('The human-readable name for this organization. This can be used to distinguish different copies of the same organization.'),
      '#required' => TRUE,
      '#size' => 30,
    ];

    $form['name'] = [
      '#type' => 'machine_name',
      '#default_value' => $this->entity->id(),
      '#maxlength' => EntityTypeInterface::BUNDLE_MAX_LENGTH,
      // '#disabled' => $this->entity->isLocked(),
      '#machine_name' => [
        'exists' => [$this->organizationStorage, 'load'],
      ],
      '#description' => $this->t('Unique machine-readable name: lowercase letters, numbers, and underscores only.'),
    ];
    $form['organization_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Organization ID'),
      '#default_value' => $this->entity->getOrganizationId(),
      '#required' => TRUE,
      '#description' => $this->t('The Coveo organization ID to search.'),
    ];
    $form['read_only'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Read only'),
      '#default_value' => $this->entity->isReadOnly(),
      '#description' => $this->t('Treat this organization as read only. Will disable any features that would push content to Coveo.'),
    ];
    // @todo hide this under read only as well?
    $form['auto_sync'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-sync'),
      '#default_value' => $this->entity->autoSync(),
      '#description' => $this->t('Should things like auto-sync with Coveo during save operations. Enabling will avoid fields getting out of sync with Coveo but this could cause additional indexing operations if fields are synced in an inconsistent state.'),
    ];
    $current_key = $this->entity->getPushKey();
    $form['push_source'] = [
      '#type' => 'details',
      '#title' => $this->t('Push source'),
    ];
    $form['push_source']['push_source_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Push source ID'),
      '#default_value' => $this->entity->getPushSourceId(),
      '#description' => $this->t('Source id associated with the push source key. Used for some API calls.'),
    ];
    $form['push_source']['push_key'] = [
      '#type' => 'password',
      '#title' => $this->t('Push API Key'),
      // @todo Communicate this is populated.
      '#description' => $this->t('The API key used for pushing content to your source. (Only enter when creating/changing)'),
      '#default_value' => '',
      '#required' => empty($current_key),
      '#size' => 60,
      '#maxlength' => 128,
      '#previous_value' => $current_key,
      '#states' => [
        'visible' => [
          'input[name="read_only"]' => ['!checked' => TRUE],
        ],
      ],
    ];
    $form['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Environment prefix'),
      '#default_value' => $this->entity->getPrefix(),
      '#description' => $this->t('Allows prefixing fields and other elements of your integration so you can create namespaces or sub-environments within your organization. This may be needed to create testing or dev environments within you non-prod environment.'),
    ];

    return parent::form($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();

    if (empty($values['push_key']) && !empty($form['push_key']['#previous_value'])) {
      $form_state->setValue('push_key', $form['push_key']['#previous_value']);
    }
    parent::validateForm(
      $form,
      $form_state
    );
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $return = parent::save($form, $form_state);
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $return;
  }

}
