<?php

declare(strict_types=1);

namespace Drupal\coveo\Form\Organization;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\coveo\Entity\CoveoOrganizationInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Sync Coveo Organization confirmation form.
 */
class CoveoOrganizationSyncForm extends ConfirmFormBase {

  /**
   * Related organization.
   */
  private CoveoOrganizationInterface $organization;

  /**
   * {@inheritDoc}
   */
  public function getFormId(): string {
    return 'coveo_organization_field_sync';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Sync fields for organization %organization?', [
      '%organization' => $this->organization->label(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('entity.coveo_organization.collection');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?CoveoOrganizationInterface $coveo_organization = NULL): array {
    if (!$coveo_organization) {
      throw new NotFoundHttpException();
    }
    if ($coveo_organization->isReadOnly()) {
      // Maybe a better error?
      throw new NotFoundHttpException();
    }
    $this->organization = $coveo_organization;
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->organization->sync();
    $this->messenger()->addStatus($this->t('All fields synced for %organization.', ['%organization' => $this->organization->label()]));
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
