<?php

declare(strict_types=1);

namespace Drupal\coveo\Form\SearchComponents;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\coveo\Entity\CoveoOrganizationInterface;
use Drupal\coveo\Plugin\CoveoSecurityProviderManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for adding and editing Coveo Search Components.
 */
class CoveoSearchComponentForm extends EntityForm {

  /**
   * The entity being used by this form.
   *
   * @var \Drupal\coveo\Entity\CoveoSearchComponentInterface
   */
  protected $entity;

  /**
   * Constructs a base class for Coveo search add and edit forms.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $searchStorage
   *   The Coveo search entity storage.
   * @param \Drupal\coveo\Plugin\CoveoSecurityProviderManagerInterface $tokenSecurityProvider
   *   The Coveo security provider manager.
   */
  public function __construct(
    protected EntityStorageInterface $searchStorage,
    protected CoveoSecurityProviderManagerInterface $tokenSecurityProvider,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')->getStorage('coveo_search_component'),
      $container->get('plugin.manager.coveo_security_provider'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Coveo search name'),
      '#default_value' => $this->entity->label(),
      '#required' => TRUE,
    ];
    $form['name'] = [
      '#type' => 'machine_name',
      '#machine_name' => [
        'exists' => [$this->searchStorage, 'load'],
      ],
      '#default_value' => $this->entity->id(),
      '#required' => TRUE,
    ];
    $current_key = $this->entity->searchKey();
    $description = ($current_key ? ' <em>Existing value hidden.</em>' : '');
    $form['search_key'] = [
      '#type' => 'password',
      '#title' => 'Search access key',
      '#default_value' => '',
      '#previous_value' => $current_key,
      '#required' => empty($current_key),
      '#description' => $description,
    ];
    $org = $this->entity->getOrganization();
    $form['organization_name'] = [
      '#type' => 'select',
      '#title' => 'Organization',
      '#default_value' => $this->entity->getOrganizationName(),
      '#options' => array_map(
        fn(CoveoOrganizationInterface $org) => $org->label(),
        $this->getOrganizations()
      ),
      '#description' => $this->t('Your organization.'),
      '#required' => TRUE,
    ];
    if ($org) {
      $form['organization_name']['#description'] .= ' ' .
        $this->t('<a href=":manage">Manage organization</a>', [
          ':manage' => $org->toUrl('edit-form')->toString(),
        ]);
    }

    $form['security_provider'] = [
      '#type' => 'select',
      '#title' => $this->t('Security provider'),
      '#default_value' => $this->entity->getSecurityProviderId(),
      '#options' => array_map(
        fn($definition) => $definition['title'],
        $this->tokenSecurityProvider->getSortedDefinitions(),
      ),
      '#required' => TRUE,
      '#description' => $this->t('Provider that will generate tokens for this search. <a href=":manage">Manage providers</a>', [
        ':manage' => Url::fromRoute('coveo.security_providers')->toString(),
      ]),
    ];

    return parent::form($form, $form_state);
  }

  /**
   * Retrieves all available Coveo organizations.
   *
   * @return \Drupal\coveo\Entity\CoveoOrganizationInterface[]
   *   The available organizations.
   */
  private function getOrganizations(): array {
    try {
      return $this->entityTypeManager
        ->getStorage('coveo_organization')
        ->loadMultiple();
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException) {
      // This should never happen.
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();

    if (empty($values['search_key']) && !empty($form['search_key']['#previous_value'])) {
      $form_state->setValue('search_key', $form['search_key']['#previous_value']);
    }
    parent::validateForm(
      $form,
      $form_state
    );
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $return = parent::save($form, $form_state);
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $return;
  }

}
