<?php

namespace Drupal\Tests\coveo\Unit;

use Drupal\coveo\FieldConverter;
use Drupal\Tests\UnitTestCase;

/**
 * Field convert tests.
 */
class FieldConverterTest extends UnitTestCase {

  /**
   * System under test.
   */
  private FieldConverter $sot;

  /**
   * {@inheritDoc}
   */
  public function setUp(): void {
    parent::setUp();
    $this->sot = new FieldConverter('foobar');
  }

  /**
   * @covers \Drupal\coveo\FieldConverter::getPrefix
   */
  public function testGetPrefix() {
    $this->assertSame('foobar', $this->sot->getPrefix());
    $this->assertSame('drpl', (new FieldConverter(''))->getPrefix());
  }

  /**
   * @covers \Drupal\coveo\FieldConverter::isDrupalField
   */
  public function testIsDrupalField() {
    $this->assertTrue($this->sot->isDrupalField('foobar_test'));
    $this->assertFalse($this->sot->isDrupalField('coveo_test'));
    $this->assertFalse($this->sot->isDrupalField('drpl_test'));
  }

  /**
   * @covers \Drupal\coveo\FieldConverter::convertCoveoToDrupal
   */
  public function testConvertCoveoToDrupal() {
    $this->assertSame('test', $this->sot->convertCoveoToDrupal('foobar_test'));
  }

  /**
   * @covers \Drupal\coveo\FieldConverter::convertDrupalToCoveo
   */
  public function testConvertDrupalToCoveo() {
    $this->assertSame('foobar_test', $this->sot->convertDrupalToCoveo('test'));
  }

  /**
   * @covers \Drupal\coveo\FieldConverter::isDrupalSystemField
   */
  public function testIsDrupalSystemField() {
    $this->assertTrue($this->sot->isDrupalSystemField(FieldConverter::COVEO_ITEM_ID_FIELD));
    $this->assertTrue($this->sot->isDrupalSystemField(FieldConverter::COVEO_PREFIX_FIELD));
    $this->assertTrue($this->sot->isDrupalSystemField(FieldConverter::COVEO_INDEX_ID_FIELD));
    $this->assertFalse($this->sot->isDrupalSystemField('coveo_test'));
  }

  /**
   * @covers \Drupal\coveo\FieldConverter::isCoveoMagic
   */
  public function testIsCoveoMagic() {
    $this->assertTrue($this->sot->isCoveoMagic('coveo_test'));
    $this->assertTrue($this->sot->isCoveoMagic('coveo_this_is_a_field'));
    $this->assertFalse($this->sot->isCoveoMagic(FieldConverter::COVEO_INDEX_ID_FIELD));
    $this->assertFalse($this->sot->isCoveoMagic('foobar_test'));
  }

  /**
   * @covers \Drupal\coveo\FieldConverter::convertCoveoMagicField
   */
  public function testConvertCoveoMagicField() {
    $this->assertSame('test', $this->sot->convertCoveoMagicField('coveo_test'));
  }

  /**
   * Provide field names for field conversion tests.
   */
  public static function providerFieldConversions(): array {
    return [
      [
        FieldConverter::COVEO_INDEX_ID_FIELD,
        FieldConverter::COVEO_INDEX_ID_FIELD,
      ],
      [
        'test',
        'coveo_test',
      ],
      [
        'foobar_test',
        'test',
      ],
      [
        'foobar_test2',
        'test2',
      ],
    ];
  }

  /**
   * @covers \Drupal\coveo\FieldConverter::convertFieldNames
   */
  public function testConvertFieldNames() {
    $this->assertEquals(
      array_column(self::providerFieldConversions(), 0),
      iterator_to_array($this->sot->convertFieldNames(
        array_column(self::providerFieldConversions(), 1),
      )),
    );
  }

  /**
   * @covers \Drupal\coveo\FieldConverter::convertFieldName
   * @dataProvider providerFieldConversions
   */
  public function testConvertFieldName($expected, $field_name) {
    $this->assertEquals(
      $expected,
      $this->sot->convertFieldName($field_name),
    );
  }

}
