<?php

namespace Drupal\credential_mask\Commands;

use Consolidation\AnnotatedCommand\CommandData;
use Consolidation\AnnotatedCommand\CommandError;
use Consolidation\OutputFormatters\Options\FormatterOptions;
use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Drupal\credential_mask\SensitiveConfigManager;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drush\Commands\DrushCommands;

/**
 * Provide Drush commands for common activities.
 */
class CredentialMaskCommands extends DrushCommands {

  use StringTranslationTrait;

  /**
   * The sensitive-config manager service.
   *
   * @var \Drupal\credential_mask\SensitiveConfigManager
   */
  protected $sensitiveConfigManager;

  /**
   * The config exported to the sync storage location.
   *
   * @var \Drupal\Core\Config\StorageInterface
   */
  protected $syncStorage;

  /**
   * Construct a CredentialMaskCommands service.
   *
   * @param \Drupal\credential_mask\SensitiveConfigManager $sensitive_config_manager
   *   The sensitive-config manager service.
   * @param \Drupal\Core\Config\StorageInterface $sync_storage
   *   Config exported to the sync storage location.
   */
  public function __construct(SensitiveConfigManager $sensitive_config_manager, StorageInterface $sync_storage) {
    parent::__construct();
    $this->sensitiveConfigManager = $sensitive_config_manager;
    $this->syncStorage = $sync_storage;
  }

  /**
   * List all active configuration properties identified as sensitive.
   *
   * @command credential_mask:list
   *
   * @usage credential_mask:list
   *   List active configuration that is identified as sensitive.
   * @field-labels
   *   name: Config name
   *   key: Key
   *   exported: Is exported?
   *   masked: Is masked?
   *
   * @return \Consolidation\OutputFormatters\StructuredData\RowsOfFields
   *   Structured data to present as a table.
   */
  public function listSensitive($options = ['format' => 'table']) {
    $rows = [];
    foreach ($this->sensitiveConfigManager->listExistingSensitiveConfig() as $config => $keys) {
      foreach ($keys as $key) {
        $exported = $this->isExported($config, $key);
        $masked = ($exported && $this->isMasked($config, $key));
        $rows[] = [
          'name' => $config,
          'key' => $key,
          'exported' => $exported,
          'masked' => ($exported) ? $masked : NULL,
        ];
      }
    }
    $data = new RowsOfFields($rows);

    // Use a renderer function to highlight unmasked sensitive config, and to
    // provide human-readable values for the exported and masked properties.
    $data->addRendererFunction(
      function ($key, $cellData, FormatterOptions $options, $rowData) {
        switch ($key) {
          case 'exported':
          case 'masked':
            if ($cellData === FALSE) {
              $output = $this->t('No');
              // Trailing space used to simulate row-highlighting.
              $output .= '        ';
            }
            elseif ($cellData === TRUE) {
              $output = $this->t('Yes');
            }
            elseif ($cellData === NULL) {
              $output = $this->t('N/A');
            }
            else {
              $output = $cellData;
            }
            break;

          default:
            $output = $cellData;
        }

        // @TODO: Investigate whether there is a better way to provide row-
        // level styling.
        if ($rowData['masked'] === FALSE && $key === 'name') {
          return '<error>' . $output;
        }
        elseif ($rowData['masked'] === FALSE && $key === 'masked') {
          return $output . '</error>';
        }
        else {
          return $output;
        }
        return $cellData;
      });

    return $data;
  }

  /**
   * Show the configuration names and properties currently marked as sensitive.
   *
   * @command credential_mask:show-configuration
   * @field-labels
   *   name: Config name
   *   key: Key
   *
   * @return \Consolidation\OutputFormatters\StructuredData\RowsOfFields
   *   Structured data to present as a table.
   */
  public function showConfig($options = ['format' => 'table']) {
    $rows = [];
    $config = $this->sensitiveConfigManager->getSensitiveConfig();

    foreach ($config as $configName => $keys) {
      foreach ($keys as $key) {
        $rows[] = [
          'name' => $configName,
          'key' => $key,
        ];
      }
    }
    return new RowsOfFields($rows);
  }

  /**
   * Mark a config key as sensitive.
   *
   * @param string $configName
   *   The name of the config. May include "*" as a wildcard.
   * @param string $configKey
   *   The config key to mask. The "." character can be used to identify an
   *   array subkey.
   *
   * @command credential_mask:add
   *
   * @usage credential_mask:add mymodule.settings apikey
   */
  public function add($configName, $configKey) {
    // This module cannot be marked as sensitive.
    if ($configName === SensitiveConfigManager::SETTINGS_KEY) {
      throw new \Exception('The credential_mask module can not be marked as sensitive.');
    }
    $this->sensitiveConfigManager->addSensitiveConfig($configName, $configKey);
  }

  /**
   * Remove a config key from the list of masked credentials.
   *
   * @param string $configName
   *   The name of the config. May include "*" as a wildcard.
   * @param string $configKey
   *   The config key to mask. The "." character can be used to identify an
   *   array subkey.
   *
   * @command credential_mask:del
   *
   * @usage credential_mask:del mymodule.settings apikey
   */
  public function delete($configName, $configKey) {
    $this->sensitiveConfigManager->deleteSensitiveConfig($configName, $configKey);
  }

  /**
   * Check whether a particular configuration is exported.
   *
   * @param string $configName
   *   The name of the configuration item: for example "search_server.settings".
   * @param string $key
   *   The configuration property, for example: "username".
   *
   * @return bool
   *   TRUE if the configuration is exported to config-sync.
   */
  protected function isExported($configName, $key) {
    if ($config = $this->syncStorage->read($configName)) {
      $parts = explode('.', $key);
      return NestedArray::KeyExists($config, $parts);
    }
    return FALSE;
  }

  /**
   * Check whether a particular configuration is exported.
   *
   * @param string $configName
   *   The name of the configuration item: for example "search_server.settings".
   * @param string $key
   *   The configuration property, for example: "username".
   *
   * @return bool
   *   TRUE if the configuration is masked in config-sync.
   */
  protected function isMasked($configName, $key) {
    if ($config = $this->syncStorage->read($configName)) {
      $parts = explode('.', $key);
      if (!NestedArray::KeyExists($config, $parts)) {
        throw new \Exception('Configuration key is not exported.');
      }
      return (NestedArray::GetValue($config, $parts) === SensitiveConfigManager::MASKING_STRING);
    }
    else {
      throw new \Exception('Configuration is not exported.');
    }
  }

  /**
   * When config:import or config:export is called, validate Drush version.
   *
   * @hook validate config:import
   */
  public function onImportConfigValidateVersion(CommandData $commandData) {
    if (!$this->drushVersionIsValid()) {
      return new CommandError('The credential_mask module requires Drush 10 or greater to operate properly with drush config:import.');
    }
  }

  /**
   * When config:import or config:export is called, validate Drush version.
   *
   * @hook validate config:export
   */
  public function onExportConfigValidateVersion(CommandData $commandData) {
    if (!$this->drushVersionIsValid()) {
      if (!$this->drushVersionIsValid()) {
        return new CommandError('The credential_mask module requires Drush 10 or greater to mask credentials with drush config:export.');
      }
    }
  }

  /**
   * Check whether the version of Drush running is valid.
   *
   * @return bool
   *   TRUE if the version of Drush can be used with the credential_mask
   *   module.
   */
  protected function drushVersionIsValid() {
    $drush_version = (class_exists('\Drush\Drush')) ? \Drush\Drush::getMajorVersion() : 8;
    return version_compare($drush_version, 10, '>=');
  }

}
