<?php

namespace Drupal\credential_mask;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Config\StorageInterface;

/**
 * Manage the list and identification of sensitive configuration.
 */
class SensitiveConfigManager {

  /**
   * String to use as a replacement for sensitive values.
   *
   * @var string
   */
  const MASKING_STRING = '<masked>';

  /**
   * Configuration name where the list of sensitive configuration is stored.
   *
   * This configuration key is immune to this module's masking.
   *
   * @var string
   */
  const SETTINGS_KEY = 'credential_mask.sensitive_config';

  /**
   * Module configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $moduleConfig;

  /**
   * Configuration names and keys of sensitive configuration to mask.
   *
   * @var array
   */
  protected $sensitiveConfig;

  /**
   * Current active database storage.
   *
   * @var \Drupal\Core\Config\DatabaseStorage
   */
  protected $activeStorage;

  /**
   * Construct a SensitiveConfig service.
   *
   * @param \Drupal\Core\Config\ConfigFactory $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Config\StorageInterface $active_storage
   *   The current active database storage.
   */
  public function __construct(ConfigFactory $config_factory, StorageInterface $active_storage) {
    $this->moduleConfig = $config_factory->getEditable(self::SETTINGS_KEY);

    // The '.' character has special meaning in Drupal config, so for the
    // config name, the '.' character is replaced by ':' in storage.
    $newKeys = array_map(function ($key) {
      return str_replace(':', '.', $key);
    }, array_keys($this->moduleConfig->get()));
    $this->sensitiveConfig = array_combine($newKeys, $this->moduleConfig->get());

    // Ensure this module never has its own masking applied.
    if (array_key_exists(self::SETTINGS_KEY, $this->sensitiveConfig)) {
      unset($this->sensitiveConfig[self::SETTINGS_KEY]);
    }

    $this->activeStorage = $active_storage;
  }

  /**
   * Get all the sensitive config that currently exists in the active config.
   *
   * @return array
   *   Associative array of config-keys, indexed by the config-name.
   */
  public function listExistingSensitiveConfig() {
    $results = [];
    $existingSensitiveConfig = $this->getSensitiveConfiguration();

    foreach ($existingSensitiveConfig as $configName => $keys) {
      if ($this->activeStorage->exists($configName)) {
        $config = $this->activeStorage->read($configName);
        foreach ($keys as $key) {
          $parts = explode('.', $key);
          if (NestedArray::keyExists($config, $parts)) {
            $results[$configName][] = $key;
          }
        }
      }
    }
    return $results;
  }

  /**
   * Get the current module configuration list of sensitive configuration.
   *
   * @return array
   *   Array of configuration keys, indexed by the configuration name.
   */
  public function getSensitiveConfig() {
    return $this->sensitiveConfig;
  }

  /**
   * Identify a configuration property as sensitive.
   *
   * @param string $configName
   *   The name of the configuration item: for example "search_server.settings".
   * @param string $key
   *   The configuration property, for example: "username".
   */
  public function addSensitiveConfig($configName, $key) {
    // Dots have special meaning as an array separator; a colon cannot be used
    // as part of a configuration name, so is a safe replacement.
    $configName = str_replace('.', ':', $configName);

    $existing = $this->moduleConfig->get($configName) ?? [];
    $existing[] = $key;
    $this->moduleConfig->set($configName, $existing);
    $this->moduleConfig->save();
  }

  /**
   * Remove a configuration property from the masking list.
   *
   * @param string $configName
   *   The name of the configuration item: for example "search_server.settings".
   * @param string $key
   *   The configuration property, for example: "username".
   */
  public function deleteSensitiveConfig($configName, $key) {
    // Dots have special meaning as an array separator; a colon cannot be used
    // as part of a configuration name, so is a safe replacement.
    $configName = str_replace('.', ':', $configName);

    $existing = $this->moduleConfig->get($configName) ?? [];
    $index = array_search($key, $existing);
    if ($index !== FALSE) {
      unset($existing[$index]);
    }

    if ($existing) {
      $this->moduleConfig->set($configName, $existing);
    }
    else {
      $this->moduleConfig->clear($configName);
    }
    $this->moduleConfig->save();
  }

  /**
   * Mask sensitive properties of a configuration storage.
   *
   * @param \Drupal\Core\Config\StorageInterface $storage
   *   The configuration to apply masking.
   */
  public function mask(StorageInterface $storage) {
    $sensitiveConfigToMask = $this->getSensitiveConfiguration($storage);

    foreach ($sensitiveConfigToMask as $configName => $keys) {
      if ($storage->exists($configName)) {
        $config = $storage->read($configName);
        foreach ($keys as $key) {
          $parts = explode('.', $key);
          if (NestedArray::keyExists($config, $parts)) {
            NestedArray::setValue($config, $parts, self::MASKING_STRING);
          }
        }
        $storage->write($configName, $config);
      }
    }
  }

  /**
   * Restore sensitive properties from the current active storage.
   *
   * @param \Drupal\Core\Config\StorageInterface $storage
   *   The configuration to unmask.
   */
  public function unmask(StorageInterface $storage) {
    $sensitiveConfigToUnMask = $this->getSensitiveConfiguration($storage);

    foreach ($sensitiveConfigToUnMask as $configName => $keys) {
      if ($storage->exists($configName)) {
        $config = $storage->read($configName);
        foreach ($keys as $key) {
          $parts = explode('.', $key);
          if (NestedArray::keyExists($config, $parts) && NestedArray::getValue($config, $parts) === self::MASKING_STRING) {
            // Attempt to read the unmasked property from active storage.
            if ($activeConfig = $this->activeStorage->read($configName)) {
              if (NestedArray::keyExists($activeConfig, $parts)) {
                $value = NestedArray::getValue($activeConfig, $parts);
                NestedArray::setValue($config, $parts, $value);
              }
            }
          }
        }
        $storage->write($configName, $config);
      }
    }
  }

  /**
   * Get a list of the configuration items which contain sensitive values.
   *
   * The set of sensitive configuration may include wildcards – this analyses
   * the configuration available in storage, and returns the list of config
   * names which meet any of the criteria, including wildcard matching against
   * config names (but not config keys).
   *
   * @param \Drupal\Core\Config\StorageInterface $storage
   *   (optional) Configuration storage to analyse. Defaults to the current
   *   active storage.
   *
   * @return array
   *   Array of sensitive properties, indexed by the configuration name.
   */
  public function getSensitiveConfiguration(StorageInterface $storage = NULL) {
    if (is_null($storage)) {
      $storage = $this->activeStorage;
    }

    $sensitiveWildcardKeys = $this->getWildcardSensitiveConfig($storage);
    $sensitiveKeys = array_diff_key($this->sensitiveConfig, array_flip($sensitiveWildcardKeys));
    $results = array_intersect_key($this->sensitiveConfig, $sensitiveKeys);

    foreach ($sensitiveWildcardKeys as $configName => $wildcardString) {
      $results[$configName] = $this->sensitiveConfig[$wildcardString];
    }

    return $results;
  }

  /**
   * List the configured wildcard entries and their existing config names.
   *
   * @param \Drupal\Core\Config\StorageInterface $storage
   *   (optional) Configuration storage to analyse. Defaults to the current
   *   active storage.
   *
   * @return array
   *   Array of wildcard strings indexed by their config name.
   */
  public function getWildcardSensitiveConfig(StorageInterface $storage = NULL) {
    if (is_null($storage)) {
      $storage = $this->activeStorage;
    }

    $sensitiveWildcardKeys = array_filter(array_keys($this->sensitiveConfig), function ($value) {
      return (strpos($value, '*') !== FALSE);
    });

    $results = [];
    foreach ($storage->listAll() as $configName) {
      foreach ($sensitiveWildcardKeys as $wildcardString) {
        if (preg_match(self::wildcardToRegex($wildcardString), $configName)) {
          $results[$configName] = $wildcardString;
        }
      }
    }

    return $results;
  }

  /**
   * Transform a string which uses a '*' as a wildcard, to a regex pattern.
   *
   * Any other regex characters in the string will be escaped.
   *
   * @param string $wildcardString
   *   A string containing a '*' wildcard character.
   *
   * @return string
   *   A string which can be used as a regex pattern representing the wildcard.
   */
  protected static function wildcardToRegex($wildcardString) {
    $result = preg_quote($wildcardString, '/');
    $result = str_replace('\*', '.*', $result);
    return "/{$result}/";
  }

}
