<?php

namespace Drupal\credential_mask\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\credential_mask\SensitiveConfigManager;

/**
 * Credential_mask module settings form.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The sensitive-config manager service.
   *
   * @var \Drupal\credential_mask\SensitiveConfigManager
   */
  protected $sensitiveConfigManager;

  /**
   * Construct a CredentialMaskCommands service.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\credential_mask\SensitiveConfigManager $sensitive_config_manager
   *   The sensitive-config manager service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, SensitiveConfigManager $sensitive_config_manager) {
    parent::__construct($config_factory);
    $this->sensitiveConfigManager = $sensitive_config_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('credential_mask.sensitive_config_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [SensitiveConfigManager::SETTINGS_KEY];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'credential_mask_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $sensitive_config = $this->sensitiveConfigManager->getSensitiveConfig();
    $sensitive_config_value = '';
    foreach ($sensitive_config as $key => $value_entries) {
      foreach ($value_entries as $value) {
        $sensitive_config_value .= $key . '|' . $value . "\n";
      }
    }

    $description = $this->t('The credential mask module integrates with the configuration management API and ensures that the configuration keys marked as "sensitive" are not exported, and when configuration is imported, the unmasked configuration is not overridden.<br />
Here you can mark a config key as sensitive using key "config name"|"config key" pairs separated by new lines.
Examples: <ul>
<li>commerce_payment.commerce_payment_gateway.card_payment_trade_business|configuration.publishable_key</li>
<li>commerce_payment.commerce_payment_gateway.card_payment_trade_business|configuration.secret_key</li>
<li>webform.webform.*|handlers.email.settings</li>
</ul>');
    $form['description'] = [
      '#markup' => $description,
    ];

    $form['config'] = [
      '#type' => 'textarea',
      '#rows' => 20,
      '#title' => $this->t('Sensitive config setting'),
      '#default_value' => $sensitive_config_value,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $sensitive_config = $this->sensitiveConfigManager->getSensitiveConfig();

    $entered_array = $this->prepareInput($form_state->getValue('config'));

    $removed_array = [];
    $created_array = [];

    $allKeys = array_unique(array_merge(array_keys($sensitive_config), array_keys($entered_array)));
    foreach ($allKeys as $key) {
      $old_values = isset($sensitive_config[$key]) ? array_unique($sensitive_config[$key]) : [];
      $new_values = isset($entered_array[$key]) ? array_unique($entered_array[$key]) : [];

      $removed = array_values(array_diff($old_values, $new_values));
      $created = array_values(array_diff($new_values, $old_values));

      if ($removed) {
        $removed_array[$key] = $removed;
      }

      if ($created) {
        $created_array[$key] = $created;
      }
    }

    foreach ($created_array as $key => $values) {
      foreach ($values as $value) {
        $this->sensitiveConfigManager->addSensitiveConfig($key, $value);
      }
    }
    foreach ($removed_array as $key => $values) {
      foreach ($values as $value) {
        $this->sensitiveConfigManager->deleteSensitiveConfig($key, $value);
      }
    }

    $this->messenger()->addMessage($this->t('The sensitive config setting has been updated.'));
  }

  /**
   * Prepare the user input into an array.
   *
   * @param string $input
   *   The user input.
   *
   * @return array
   *   The prepared array.
   */
  private function prepareInput($input) {
    $value = explode("\n", str_replace("\r", "\n", $input));
    $values = array_unique(array_filter(array_map('trim', $value)));

    $items = [];
    foreach ($values as $row) {
      if ($this->isInvalid($row)) {
        $this->messenger()->addError($this->t('@row is not a valid entry', [
          '@row' => $row,
        ]));
        continue;
      }

      [$config_name, $config_key] = explode('|', $row);
      $items[$config_name][] = $config_key;
    }

    return $items;
  }

  /**
   * Check if the row is invalid.
   *
   * @param string $row
   *
   * @return bool
   *   True if the row is invalid.
   */
  public function isInvalid(mixed $row): bool {
    return str_contains($row, ' ') || substr_count($row, '|') !== 1;
  }

}
