<?php

namespace Drupal\cricket_chatbot\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\cricket_chatbot\Service\CricketApiService;

/**
 * Controller for the cricket chatbot API endpoint.
 */
class ChatbotController extends ControllerBase {

  /**
   * The cricket API service.
   *
   * @var \Drupal\cricket_chatbot\Service\CricketApiService
   */
  protected $cricketApiService;

  /**
   * Constructs a new ChatbotController object.
   *
   * @param \Drupal\cricket_chatbot\Service\CricketApiService $cricket_api_service
   *   The cricket API service.
   */
  public function __construct(CricketApiService $cricket_api_service) {
    $this->cricketApiService = $cricket_api_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('cricket_chatbot.api_service')
    );
  }

  /**
   * Process a chat request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response.
   */
  public function processChat(Request $request) {
    $content = json_decode($request->getContent(), TRUE);
    $query = $content['query'] ?? '';

    if (empty($query)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'No query provided',
      ]);
    }

    // Process the query to find intent and entities
    $intent = $this->determineIntent($query);
    $entities = $this->extractEntities($query);

    // Get response based on intent and entities
    $response = $this->getResponse($intent, $entities);

    return new JsonResponse([
      'status' => 'success',
      'response' => $response,
    ]);
  }

  /**
   * Determine the intent of the user query.
   *
   * @param string $query
   *   The user query.
   *
   * @return string
   *   The determined intent.
   */
  private function determineIntent($query) {
    $query = strtolower($query);

    // Series-related queries
    if (strpos($query, 'series') !== FALSE || strpos($query, 'tournament') !== FALSE ||
        strpos($query, 'championship') !== FALSE || strpos($query, 'cup') !== FALSE ||
        strpos($query, 'trophy') !== FALSE || strpos($query, 'league') !== FALSE) {
      return 'series_info';
    }

    // Player-related queries
    if (strpos($query, 'player') !== FALSE || strpos($query, 'batsman') !== FALSE ||
        strpos($query, 'bowler') !== FALSE || strpos($query, 'all-rounder') !== FALSE ||
        strpos($query, 'captain') !== FALSE || strpos($query, 'stats') !== FALSE ||
        strpos($query, 'statistics') !== FALSE || strpos($query, 'career') !== FALSE ||
        strpos($query, 'batting') !== FALSE || strpos($query, 'bowling') !== FALSE) {
      return 'player_info';
    }

    // Match detail queries
    if (strpos($query, 'scorecard') !== FALSE || strpos($query, 'detail') !== FALSE ||
        strpos($query, 'full score') !== FALSE || strpos($query, 'innings') !== FALSE ||
        strpos($query, 'wickets') !== FALSE || strpos($query, 'runs') !== FALSE ||
        strpos($query, 'highlights') !== FALSE) {
      return 'match_detail';
    }

    // Existing intents
    if (strpos($query, 'score') !== FALSE || strpos($query, 'match') !== FALSE ||
        strpos($query, 'playing') !== FALSE || strpos($query, 'going on') !== FALSE) {
      return 'current_match';
    }

    if (strpos($query, 'next') !== FALSE || strpos($query, 'upcoming') !== FALSE ||
        strpos($query, 'scheduled') !== FALSE || strpos($query, 'future') !== FALSE) {
      return 'upcoming_match';
    }

    if (strpos($query, 'win') !== FALSE || strpos($query, 'won') !== FALSE ||
        strpos($query, 'lost') !== FALSE || strpos($query, 'result') !== FALSE ||
        strpos($query, 'yesterday') !== FALSE || strpos($query, 'last') !== FALSE) {
      return 'past_match';
    }

    return 'unknown';
  }

  /**
   * Extract entities from the query.
   *
   * @param string $query
   *   The user query.
   *
   * @return array
   *   An array of entities.
   */
  private function extractEntities($query) {
    $entities = [
      'teams' => $this->extractTeams($query),
      'players' => $this->extractPlayers($query),
      'series' => $this->extractSeries($query),
      'match_id' => NULL,
    ];

    return $entities;
  }

  /**
   * Extract team names from the query.
   *
   * @param string $query
   *   The user query.
   *
   * @return array
   *   An array of team names.
   */
  private function extractTeams($query) {
    $teams = [];
    $query = strtolower($query);

    // Map of team names and their variations/abbreviations
    $teamMappings = [
      'india' => ['india', 'ind', 'team india', 'men in blue', 'indian'],
      'australia' => ['australia', 'aus', 'aussies', 'australian'],
      'england' => ['england', 'eng', 'english'],
      'pakistan' => ['pakistan', 'pak', 'pakistani'],
      'south africa' => ['south africa', 'sa', 'proteas'],
      'new zealand' => ['new zealand', 'nz', 'black caps', 'kiwis'],
      'west indies' => ['west indies', 'wi', 'windies', 'caribbean'],
      'sri lanka' => ['sri lanka', 'sl', 'lanka', 'lankan'],
      'bangladesh' => ['bangladesh', 'ban', 'tigers', 'bangla'],
      'afghanistan' => ['afghanistan', 'afg', 'afghan'],
      'zimbabwe' => ['zimbabwe', 'zim', 'zimbabwean'],
      'ireland' => ['ireland', 'ire', 'irish'],
      // Add IPL teams
      'mumbai indians' => ['mumbai', 'mi', 'mumbai indians'],
      'chennai super kings' => ['chennai', 'csk', 'chennai super kings', 'super kings'],
      'royal challengers bangalore' => ['bangalore', 'rcb', 'royal challengers', 'royal challengers bangalore'],
      'kolkata knight riders' => ['kolkata', 'kkr', 'knight riders', 'kolkata knight riders'],
      'delhi capitals' => ['delhi', 'dc', 'capitals', 'delhi capitals'],
      'rajasthan royals' => ['rajasthan', 'rr', 'royals', 'rajasthan royals'],
      'punjab kings' => ['punjab', 'pbks', 'kings xi', 'punjab kings'],
      'sunrisers hyderabad' => ['hyderabad', 'srh', 'sunrisers', 'sunrisers hyderabad'],
    ];

    // Check for team names in the query
    foreach ($teamMappings as $teamName => $variations) {
      foreach ($variations as $variation) {
        if (strpos($query, $variation) !== FALSE) {
          $teams[] = $teamName;
          break; // Found a match for this team, move to next team
        }
      }
    }

    return $teams;
  }

  /**
   * Extract player names from the query.
   *
   * @param string $query
   *   The user query.
   *
   * @return array
   *   An array of player names.
   */
  private function extractPlayers($query) {
    $players = [];
    $query = strtolower($query);

    // Map of player names and their variations/nicknames
    $playerMappings = [
      // Indian players
      'Virat Kohli' => ['virat', 'kohli', 'king kohli', 'vk'],
      'Rohit Sharma' => ['rohit', 'sharma', 'hitman', 'rohit sharma'],
      'MS Dhoni' => ['dhoni', 'ms', 'msd', 'ms dhoni', 'mahendra singh dhoni', 'thala'],
      'Jasprit Bumrah' => ['bumrah', 'jasprit', 'boom', 'boom boom bumrah'],
      'KL Rahul' => ['rahul', 'kl', 'kl rahul', 'lokesh'],
      'Ravindra Jadeja' => ['jadeja', 'ravindra', 'jaddu', 'sir jadeja'],

      // Australian players
      'Steve Smith' => ['smith', 'steve', 'smudge', 'steve smith'],
      'David Warner' => ['warner', 'david', 'dave', 'david warner'],
      'Pat Cummins' => ['cummins', 'pat', 'patrick', 'pat cummins'],
      'Mitchell Starc' => ['starc', 'mitchell', 'mitch starc', 'mitchell starc'],
      'Glenn Maxwell' => ['maxwell', 'glenn', 'maxi', 'the big show', 'glenn maxwell'],

      // English players
      'Joe Root' => ['root', 'joe', 'joe root', 'rooty'],
      'Ben Stokes' => ['stokes', 'ben', 'ben stokes'],
      'James Anderson' => ['anderson', 'james', 'jimmy', 'james anderson'],
      'Jos Buttler' => ['buttler', 'jos', 'jos buttler'],
      'Stuart Broad' => ['broad', 'stuart', 'stuart broad'],

      // Other international players
      'Kane Williamson' => ['williamson', 'kane', 'kane williamson'],
      'Babar Azam' => ['babar', 'azam', 'babar azam'],
      'Quinton de Kock' => ['de kock', 'quinton', 'qdk', 'quinton de kock'],
      'Shakib Al Hasan' => ['shakib', 'al hasan', 'shakib al hasan'],
      'Trent Boult' => ['boult', 'trent', 'trent boult'],
      'Rashid Khan' => ['rashid', 'khan', 'rashid khan'],
      'Kagiso Rabada' => ['rabada', 'kagiso', 'kg', 'kagiso rabada'],

      // Retired legends (still popular in queries)
      'Sachin Tendulkar' => ['sachin', 'tendulkar', 'master blaster', 'little master', 'god of cricket'],
      'Brian Lara' => ['lara', 'brian', 'prince', 'brian lara'],
      'Shane Warne' => ['warne', 'shane', 'warnie', 'shane warne'],
      'Ricky Ponting' => ['ponting', 'ricky', 'punter', 'ricky ponting'],
      'Jacques Kallis' => ['kallis', 'jacques', 'jacques kallis'],
      'AB de Villiers' => ['de villiers', 'ab', 'abcv', 'abd', 'mr 360', 'ab de villiers'],
      'Kumar Sangakkara' => ['sangakkara', 'kumar', 'sanga', 'kumar sangakkara'],
    ];

    // Check for player names in the query
    foreach ($playerMappings as $playerName => $variations) {
      foreach ($variations as $variation) {
        if (strpos($query, $variation) !== FALSE) {
          $players[] = $playerName;
          break; // Found a match for this player, move to next player
        }
      }
    }

    return $players;
  }

  /**
   * Extract series names from the query.
   *
   * @param string $query
   *   The user query.
   *
   * @return array
   *   An array of series names.
   */
  private function extractSeries($query) {
    $series = [];
    $query = strtolower($query);

    // Map of series names and their variations/abbreviations
    $seriesMappings = [
      // ICC tournaments
      'ICC Cricket World Cup' => ['world cup', 'cricket world cup', 'cwc', 'odi world cup'],
      'ICC T20 World Cup' => ['t20 world cup', 't20wc', 't20 wc', 'twenty20 world cup'],
      'ICC Champions Trophy' => ['champions trophy', 'ct'],
      'ICC World Test Championship' => ['world test championship', 'wtc', 'test championship'],

      // Major leagues
      'Indian Premier League' => ['ipl', 'indian premier league'],
      'Big Bash League' => ['big bash', 'bbl', 'big bash league'],
      'Pakistan Super League' => ['psl', 'pakistan super league'],
      'Caribbean Premier League' => ['cpl', 'caribbean premier league'],
      'Bangladesh Premier League' => ['bpl', 'bangladesh premier league'],
      'The Hundred' => ['the hundred', 'hundred'],
      'Vitality Blast' => ['vitality blast', 't20 blast'],

      // Famous bilateral series
      'The Ashes' => ['ashes', 'the ashes'],
      'Border-Gavaskar Trophy' => ['border-gavaskar', 'bgt', 'border gavaskar trophy', 'border-gavaskar trophy'],
      'Basil D\'Oliveira Trophy' => ['basil d\'oliveira', 'oliveira trophy', 'd\'oliveira'],
      'Freedom Trophy' => ['freedom trophy', 'freedom series'],
      'Warne-Muralitharan Trophy' => ['warne-muralitharan', 'warne muralitharan trophy'],

      // Generic series types
      'Test Series' => ['test series', 'test matches'],
      'ODI Series' => ['odi series', 'one day series', 'one-day series', 'odis'],
      'T20I Series' => ['t20i series', 't20 series', 't20 internationals', 't20is'],
    ];

    // Check for series names in the query
    foreach ($seriesMappings as $seriesName => $variations) {
      foreach ($variations as $variation) {
        if (strpos($query, $variation) !== FALSE) {
          $series[] = $seriesName;
          break; // Found a match for this series, move to next series
        }
      }
    }

    // Look for bilateral series patterns (e.g., "India vs Australia series")
    if (empty($series) && (strpos($query, ' vs ') !== FALSE || strpos($query, ' versus ') !== FALSE || strpos($query, '-') !== FALSE)) {
      $teams = $this->extractTeams($query);

      if (count($teams) >= 2) {
        // Create a generic bilateral series name
        $series[] = $teams[0] . ' vs ' . $teams[1] . ' Series';
      }
    }

    return $series;
  }

  /**
   * Get a response based on intent and entities.
   *
   * @param string $intent
   *   The determined intent.
   * @param array $entities
   *   The extracted entities.
   *
   * @return string
   *   The chatbot response.
   */
  private function getResponse($intent, $entities) {
    switch ($intent) {
      case 'series_info':
        return $this->getSeriesInfoResponse($entities['series']);

      case 'player_info':
        return $this->getPlayerInfoResponse($entities['players']);

      case 'match_detail':
        return $this->getMatchDetailResponse($entities['match_id'], $entities['teams']);

      case 'current_match':
        return $this->getCurrentMatchResponse($entities['teams']);

      case 'upcoming_match':
        return $this->getUpcomingMatchResponse($entities['teams']);

      case 'past_match':
        return $this->getPastMatchResponse($entities['teams']);

      default:
        return "I'm sorry, I didn't understand your question. You can ask me about current matches, upcoming matches, recent results, series information, or player stats. For example, 'What's the score in the India match?', 'Tell me about the IPL', or 'Show me Virat Kohli's stats'.";
    }
  }

  /**
   * Get response for series info queries.
   *
   * @param array $series
   *   The series to check for.
   *
   * @return string
   *   The response.
   */
  private function getSeriesInfoResponse($series) {
    try {
      // If specific series were mentioned, get info for the first one
      $series_name = !empty($series) ? $series[0] : NULL;

      // For demo purposes, we'll use a generic response
      if ($series_name) {
        $series_info = $this->cricketApiService->getSeriesInfo($series_name);

        if (empty($series_info)) {
          return "I couldn't find information about the {$series_name}. Please try asking about a different tournament.";
        }

        $response = "Information about {$series_name}:\n\n";

        if (!empty($series_info['startDate']) && !empty($series_info['endDate'])) {
          $response .= "Duration: {$series_info['startDate']} to {$series_info['endDate']}\n\n";
        }

        // Add standings if available
        if (!empty($series_info['standings'])) {
          $response .= "Current Standings:\n";

          foreach ($series_info['standings'] as $standing) {
            $response .= "{$standing['position']}. {$standing['team']} - {$standing['points']} points";
            $response .= " (W: {$standing['won']}, L: {$standing['lost']}, NRR: {$standing['nrr']})\n";
          }

          $response .= "\n";
        }

        // Add recent/upcoming matches if available
        if (!empty($series_info['matches'])) {
          $response .= "Matches:\n";

          $count = 0;
          foreach ($series_info['matches'] as $match) {
            $response .= "{$match['name']} - {$match['date']}\n";
            $response .= "Status: {$match['status']}\n";
            $response .= "Venue: {$match['venue']}\n\n";

            $count++;
            if ($count >= 3) {
              break; // Limit to 3 matches to keep response concise
            }
          }
        }

        return $response;
      }
      else {
        // Get information about current/upcoming series
        $series_info = $this->cricketApiService->getSeriesInfo();

        if (empty($series_info)) {
          return "I couldn't find information about any current cricket series. Please try asking about a specific tournament like the IPL or World Cup.";
        }

        $response = "Current Cricket Series:\n\n";

        if (is_array($series_info)) {
          // If multiple series are returned
          foreach ($series_info as $series) {
            $response .= "{$series['name']}\n";

            if (!empty($series['startDate']) && !empty($series['endDate'])) {
              $response .= "Duration: {$series['startDate']} to {$series['endDate']}\n";
            }

            $response .= "\n";
          }
        }
        else {
          // Single series details
          $response .= "{$series_info['name']}\n";

          if (!empty($series_info['startDate']) && !empty($series_info['endDate'])) {
            $response .= "Duration: {$series_info['startDate']} to {$series_info['endDate']}\n\n";
          }

          // Add recent/upcoming matches if available
          if (!empty($series_info['matches'])) {
            $response .= "Recent/Upcoming Matches:\n";

            $count = 0;
            foreach ($series_info['matches'] as $match) {
              $response .= "{$match['name']} - {$match['date']}\n";
              $response .= "Status: {$match['status']}\n";
              $response .= "Venue: {$match['venue']}\n\n";

              $count++;
              if ($count >= 3) {
                break; // Limit to 3 matches to keep response concise
              }
            }
          }
        }

        return $response;
      }
    }
    catch (\Exception $e) {
      return "Sorry, I couldn't retrieve the series information. Please try again later.";
    }
  }

  /**
   * Get response for player info queries.
   *
   * @param array $players
   *   The players to check for.
   *
   * @return string
   *   The response.
   */
  private function getPlayerInfoResponse($players) {
    try {
      // If specific players were mentioned, get info for the first one
      if (!empty($players)) {
        $player_name = $players[0];
        $player_info = $this->cricketApiService->getPlayerInfo($player_name);

        if (empty($player_info)) {
          return "I couldn't find information about {$player_name}. Please try asking about a different player.";
        }

        $response = "Information about {$player_info['name']}:\n\n";

        if (!empty($player_info['country'])) {
          $response .= "Country: {$player_info['country']}\n";
        }

        if (!empty($player_info['role'])) {
          $response .= "Role: {$player_info['role']}\n";
        }

        if (!empty($player_info['battingStyle'])) {
          $response .= "Batting Style: {$player_info['battingStyle']}\n";
        }

        if (!empty($player_info['bowlingStyle'])) {
          $response .= "Bowling Style: {$player_info['bowlingStyle']}\n";
        }

        $response .= "\n";

        // Add batting stats if available
        if (!empty($player_info['stats']['batting'])) {
          $response .= "Batting Statistics:\n";

          foreach ($player_info['stats']['batting'] as $format => $stats) {
            $response .= "{$format}:\n";
            $response .= "Matches: {$stats['matches']}, Innings: {$stats['innings']}, Runs: {$stats['runs']}\n";
            $response .= "Average: {$stats['average']}, Strike Rate: {$stats['strikeRate']}\n";
            $response .= "Highest Score: {$stats['highestScore']}, 100s: {$stats['centuries']}, 50s: {$stats['halfCenturies']}\n\n";
          }
        }

        // Add bowling stats if available
        if (!empty($player_info['stats']['bowling'])) {
          $response .= "Bowling Statistics:\n";

          foreach ($player_info['stats']['bowling'] as $format => $stats) {
            $response .= "{$format}:\n";
            $response .= "Matches: {$stats['matches']}, Innings: {$stats['innings']}, Wickets: {$stats['wickets']}\n";
            $response .= "Economy: {$stats['economy']}, Average: {$stats['average']}, Strike Rate: {$stats['strikeRate']}\n";
            $response .= "Best Innings: {$stats['bestInnings']}, 5 Wickets: {$stats['fiveWickets']}\n\n";
          }
        }

        return $response;
      }
      else {
        return "Please specify a player name. For example, 'Tell me about Virat Kohli's stats' or 'Show me information about Joe Root'.";
      }
    }
    catch (\Exception $e) {
      return "Sorry, I couldn't retrieve the player information. Please try again later.";
    }
  }

  /**
   * Get response for match detail queries.
   *
   * @param string|null $match_id
   *   The match ID to get details for.
   * @param array $teams
   *   The teams to check for if match ID is not available.
   *
   * @return string
   *   The response.
   */
  private function getMatchDetailResponse($match_id = NULL, array $teams = []) {
    try {
      // If match ID is provided, get details for that match
      if ($match_id) {
        $match_info = $this->cricketApiService->getMatchInfo($match_id);
      }
      else {
        // Otherwise, get current matches and find one with the specified teams
        $matches = $this->cricketApiService->getCurrentMatches();

        if (empty($matches)) {
          return "There are no matches currently in progress. Please try asking about a specific match.";
        }

        // If teams were specified, filter matches
        if (!empty($teams)) {
          $team_matches = [];
          foreach ($matches as $match) {
            foreach ($teams as $team) {
              if (stripos($match['team1'], $team) !== FALSE || stripos($match['team2'], $team) !== FALSE) {
                $team_matches[] = $match;
                break;
              }
            }
          }

          if (empty($team_matches)) {
            return "There are no current matches for " . implode(' or ', $teams) . " at the moment.";
          }

          // Get details for the first matching match
          $match_info = $this->cricketApiService->getMatchInfo($team_matches[0]['id']);
        }
        else {
          // Get details for the first current match
          $match_info = $this->cricketApiService->getMatchInfo($matches[0]['id']);
        }
      }

      if (empty($match_info)) {
        return "I couldn't find detailed information for this match. Please try again later.";
      }

      // Format response
      $response = "Match Details: {$match_info['name']}\n\n";
      $response .= "Status: {$match_info['status']}\n";
      $response .= "Venue: {$match_info['venue']}\n";
      $response .= "Date: {$match_info['date']}\n\n";

      if (!empty($match_info['tossWinner'])) {
        $response .= "Toss: {$match_info['tossWinner']} won and chose to {$match_info['tossChoice']}\n\n";
      }

      // Add scores if available
      if (!empty($match_info['score'])) {
        $response .= "Current Score:\n";

        foreach ($match_info['score'] as $score) {
          $inning = explode('/', $score['team']);
          $team = $inning[0] ?? '';
          $response .= "{$team}: {$score['runs']}/{$score['wickets']} ({$score['overs']} overs)\n";
        }

        $response .= "\n";
      }

      // Add scorecard summary if available
      if (!empty($match_info['scorecard'])) {
        $response .= "Scorecard Highlights:\n\n";

        foreach ($match_info['scorecard'] as $inning) {
          $response .= "{$inning['inning']}:\n";

          // Add top batsmen
          if (!empty($inning['batting'])) {
            $response .= "Top Batsmen:\n";

            $count = 0;
            usort($inning['batting'], function($a, $b) {
              return $b['runs'] - $a['runs'];
            });

            foreach ($inning['batting'] as $batsman) {
              $response .= "{$batsman['name']}: {$batsman['runs']} ({$batsman['balls']} balls, {$batsman['fours']} 4s, {$batsman['sixes']} 6s)\n";

              $count++;
              if ($count >= 3) {
                break; // Limit to top 3 batsmen
              }
            }

            $response .= "\n";
          }

          // Add top bowlers
          if (!empty($inning['bowling'])) {
            $response .= "Top Bowlers:\n";

            $count = 0;
            usort($inning['bowling'], function($a, $b) {
              return $b['wickets'] - $a['wickets'];
            });

            foreach ($inning['bowling'] as $bowler) {
              $response .= "{$bowler['name']}: {$bowler['wickets']}/{$bowler['runs']} ({$bowler['overs']} overs)\n";

              $count++;
              if ($count >= 3) {
                break; // Limit to top 3 bowlers
              }
            }

            $response .= "\n";
          }
        }
      }

      if (!empty($match_info['matchWinner'])) {
        $response .= "Result: {$match_info['matchWinner']} won\n";
      }

      return $response;
    }
    catch (\Exception $e) {
      return "Sorry, I couldn't retrieve the match details. Please try again later.";
    }
  }

  /**
   * Get response for current match queries.
   *
   * @param array $teams
   *   The teams to check for.
   *
   * @return string
   *   The response.
   */
  private function getCurrentMatchResponse($teams) {
    try {
      $matches = $this->cricketApiService->getCurrentMatches();

      if (empty($matches)) {
        return "There are no matches currently in progress.";
      }

      // If specific teams were mentioned, filter matches
      if (!empty($teams)) {
        $teamMatches = [];
        foreach ($matches as $match) {
          foreach ($teams as $team) {
            if (stripos($match['team1'], $team) !== FALSE || stripos($match['team2'], $team) !== FALSE) {
              $teamMatches[] = $match;
              break;
            }
          }
        }

        if (empty($teamMatches)) {
          return "There are no current matches for " . implode(' or ', $teams) . " at the moment.";
        }

        $matches = $teamMatches;
      }

      // Format response
      $response = "Current match";
      if (count($matches) > 1) {
        $response .= "es";
      }
      $response .= ":\n\n";

      foreach ($matches as $match) {
        $response .= "{$match['team1']} vs {$match['team2']}\n";
        $response .= "Status: {$match['status']}\n";
        if (isset($match['score'])) {
          $response .= "Score: {$match['score']}\n";
        }
        $response .= "\n";
      }

      return $response;
    }
    catch (\Exception $e) {
      return "Sorry, I couldn't retrieve the current match information. Please try again later.";
    }
  }

  /**
   * Get response for upcoming match queries.
   *
   * @param array $teams
   *   The teams to check for.
   *
   * @return string
   *   The response.
   */
  private function getUpcomingMatchResponse($teams) {
    try {
      $matches = $this->cricketApiService->getUpcomingMatches();

      if (empty($matches)) {
        return "There are no upcoming matches scheduled.";
      }

      // If specific teams were mentioned, filter matches
      if (!empty($teams)) {
        $teamMatches = [];
        foreach ($matches as $match) {
          foreach ($teams as $team) {
            if (stripos($match['team1'], $team) !== FALSE || stripos($match['team2'], $team) !== FALSE) {
              $teamMatches[] = $match;
              break;
            }
          }
        }

        if (empty($teamMatches)) {
          return "There are no upcoming matches scheduled for " . implode(' or ', $teams) . ".";
        }

        $matches = $teamMatches;
      }

      // Format response
      $response = "Upcoming match";
      if (count($matches) > 1) {
        $response .= "es";
      }
      $response .= ":\n\n";

      foreach ($matches as $match) {
        $response .= "{$match['team1']} vs {$match['team2']}\n";
        $response .= "Date: {$match['date']}\n";
        $response .= "Venue: {$match['venue']}\n\n";
      }

      return $response;
    }
    catch (\Exception $e) {
      return "Sorry, I couldn't retrieve the upcoming match information. Please try again later.";
    }
  }

  /**
   * Get response for past match queries.
   *
   * @param array $teams
   *   The teams to check for.
   *
   * @return string
   *   The response.
   */
  private function getPastMatchResponse($teams) {
    try {
      $matches = $this->cricketApiService->getRecentMatches();

      if (empty($matches)) {
        return "There are no recent match results available.";
      }

      // If specific teams were mentioned, filter matches
      if (!empty($teams)) {
        $teamMatches = [];
        foreach ($matches as $match) {
          foreach ($teams as $team) {
            if (stripos($match['team1'], $team) !== FALSE || stripos($match['team2'], $team) !== FALSE) {
              $teamMatches[] = $match;
              break;
            }
          }
        }

        if (empty($teamMatches)) {
          return "There are no recent match results for " . implode(' or ', $teams) . ".";
        }

        $matches = $teamMatches;
      }

      // Format response
      $response = "Recent match result";
      if (count($matches) > 1) {
        $response .= "s";
      }
      $response .= ":\n\n";

      foreach ($matches as $match) {
        $response .= "{$match['team1']} vs {$match['team2']}\n";
        $response .= "Result: {$match['result']}\n\n";
      }

      return $response;
    }
    catch (\Exception $e) {
      return "Sorry, I couldn't retrieve the recent match results. Please try again later.";
    }
  }

}
