<?php

namespace Drupal\cricket_chatbot\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Cricket Chatbot settings for this site.
 */
class ConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'cricket_chatbot_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['cricket_chatbot.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('cricket_chatbot.settings');

    $form['api_provider'] = [
      '#type' => 'select',
      '#title' => $this->t('API Provider'),
      '#default_value' => $config->get('api_provider') ?? 'cricket_data_org',
      '#options' => [
        'cricket_data_org' => $this->t('Cricket Data.org (Free)'),
        'api_cricket' => $this->t('API-Cricket via RapidAPI (Free tier)'),
        'api_sports' => $this->t('API-Sports Cricket via RapidAPI (Free tier)'),
      ],
      '#description' => $this->t('Select which cricket data API to use. All options include a free tier with limited requests.'),
      '#required' => TRUE,
    ];

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#default_value' => $config->get('api_key') ?? '',
      '#description' => $this->t('Enter your API key for the selected provider. For Cricket Data.org, you can leave this empty for the free tier.'),
    ];

    $form['cache_time'] = [
      '#type' => 'number',
      '#title' => $this->t('Cache Time'),
      '#default_value' => $config->get('cache_time') ?? 600,
      '#min' => 60,
      '#description' => $this->t('Time in seconds to cache API responses. Minimum is 60 seconds. Longer cache times help stay within API rate limits.'),
      '#required' => TRUE,
    ];

    $form['welcome_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Welcome Message'),
      '#default_value' => $config->get('welcome_message') ?? 'Hi there! I can help you with cricket scores and match information. Ask me about current matches, upcoming games, or recent results!',
      '#description' => $this->t('Welcome message to show when the chatbot first loads.'),
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $api_provider = $form_state->getValue('api_provider');
    $api_key = $form_state->getValue('api_key');

    // RapidAPI providers require an API key
    if (($api_provider === 'api_cricket' || $api_provider === 'api_sports') && empty($api_key)) {
      $form_state->setErrorByName('api_key', $this->t('API key is required for RapidAPI providers.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('cricket_chatbot.settings')
      ->set('api_provider', $form_state->getValue('api_provider'))
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('cache_time', $form_state->getValue('cache_time'))
      ->set('welcome_message', $form_state->getValue('welcome_message'))
      ->save();

    parent::submitForm($form, $form_state);
  }
}
