<?php

namespace Drupal\cricket_chatbot\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Provides a 'Cricket Chatbot' block.
 *
 * @Block(
 *   id = "cricket_chatbot_block",
 *   admin_label = @Translation("Cricket Chatbot"),
 *   category = @Translation("Custom")
 * )
 */
class ChatbotBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new ChatbotBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'chatbot_title' => $this->t('Cricket Chatbot'),
      'height' => 400,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);
    $config = $this->getConfiguration();

    $form['chatbot_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Chatbot Title'),
      '#default_value' => $config['chatbot_title'],
      '#description' => $this->t('Title to display above the chatbot.'),
      '#required' => TRUE,
    ];

    $form['height'] = [
      '#type' => 'number',
      '#title' => $this->t('Chatbot Height'),
      '#default_value' => $config['height'],
      '#description' => $this->t('Height of the chatbot in pixels.'),
      '#min' => 200,
      '#max' => 800,
      '#step' => 50,
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $values = $form_state->getValues();
    $this->configuration['chatbot_title'] = $values['chatbot_title'];
    $this->configuration['height'] = $values['height'];
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $config = $this->getConfiguration();
    $moduleConfig = $this->configFactory->get('cricket_chatbot.settings');

    $chatbot_id = 'cricket-chatbot-' . $this->getPluginId();

    $build = [
      '#theme' => 'cricket_chatbot_block',
      '#chatbot_id' => $chatbot_id,
      '#welcome_message' => $moduleConfig->get('welcome_message'),
      '#attached' => [
        'library' => ['cricket_chatbot/chatbot'],
        'drupalSettings' => [
          'cricketChatbot' => [
            $chatbot_id => [
              'height' => $config['height'],
              'title' => $config['chatbot_title'],
            ],
          ],
        ],
      ],
    ];

    return $build;
  }

}
