<?php

namespace Drupal\cricket_chatbot\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Service for interacting with cricket APIs.
 */
class CricketApiService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a new CricketApiService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ClientInterface $http_client,
    CacheBackendInterface $cache,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
    $this->cache = $cache;
    $this->logger = $logger_factory->get('cricket_chatbot');
  }

  /**
   * Get current matches.
   *
   * @return array
   *   An array of current matches.
   */
  public function getCurrentMatches() {
    $cache_id = 'cricket_chatbot:current_matches';

    if ($cache = $this->cache->get($cache_id)) {
      return $cache->data;
    }

    $config = $this->configFactory->get('cricket_chatbot.settings');
    $provider = $config->get('api_provider') ?? 'cricket_data_org';
    $api_key = $config->get('api_key');
    $cache_time = $config->get('cache_time') ?? 600;

    try {
      switch ($provider) {
        case 'cricket_data_org':
          $matches = $this->fetchCricketDataOrgMatches('current');
          break;

        case 'api_cricket':
          $matches = $this->fetchApiCricketMatches('current', $api_key);
          break;

        case 'api_sports':
          $matches = $this->fetchApiSportsMatches('current', $api_key);
          break;

        default:
          $matches = [];
      }

      $this->cache->set($cache_id, $matches, time() + $cache_time);
      return $matches;
    }
    catch (\Exception $e) {
      $this->logger->error('Error fetching current matches: @message', ['@message' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * Get upcoming matches.
   *
   * @return array
   *   An array of upcoming matches.
   */
  public function getUpcomingMatches() {
    $cache_id = 'cricket_chatbot:upcoming_matches';

    if ($cache = $this->cache->get($cache_id)) {
      return $cache->data;
    }

    $config = $this->configFactory->get('cricket_chatbot.settings');
    $provider = $config->get('api_provider') ?? 'cricket_data_org';
    $api_key = $config->get('api_key');
    $cache_time = $config->get('cache_time') ?? 600;

    try {
      switch ($provider) {
        case 'cricket_data_org':
          $matches = $this->fetchCricketDataOrgMatches('upcoming');
          break;

        case 'api_cricket':
          $matches = $this->fetchApiCricketMatches('upcoming', $api_key);
          break;

        case 'api_sports':
          $matches = $this->fetchApiSportsMatches('upcoming', $api_key);
          break;

        default:
          $matches = [];
      }

      $this->cache->set($cache_id, $matches, time() + $cache_time);
      return $matches;
    }
    catch (\Exception $e) {
      $this->logger->error('Error fetching upcoming matches: @message', ['@message' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * Get recent matches.
   *
   * @return array
   *   An array of recent matches.
   */
  public function getRecentMatches() {
    $cache_id = 'cricket_chatbot:recent_matches';

    if ($cache = $this->cache->get($cache_id)) {
      return $cache->data;
    }

    $config = $this->configFactory->get('cricket_chatbot.settings');
    $provider = $config->get('api_provider') ?? 'cricket_data_org';
    $api_key = $config->get('api_key');
    $cache_time = $config->get('cache_time') ?? 600;

    try {
      switch ($provider) {
        case 'cricket_data_org':
          $matches = $this->fetchCricketDataOrgMatches('recent');
          break;

        case 'api_cricket':
          $matches = $this->fetchApiCricketMatches('recent', $api_key);
          break;

        case 'api_sports':
          $matches = $this->fetchApiSportsMatches('recent', $api_key);
          break;

        default:
          $matches = [];
      }

      $this->cache->set($cache_id, $matches, time() + $cache_time);
      return $matches;
    }
    catch (\Exception $e) {
      $this->logger->error('Error fetching recent matches: @message', ['@message' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * Get information about a specific series.
   *
   * @param string $series_id
   *   The series ID to fetch information for.
   *
   * @return array
   *   Series information.
   */
  public function getSeriesInfo($series_id = NULL) {
    $cache_id = 'cricket_chatbot:series_info:' . ($series_id ?? 'current');

    if ($cache = $this->cache->get($cache_id)) {
      return $cache->data;
    }

    $config = $this->configFactory->get('cricket_chatbot.settings');
    $provider = $config->get('api_provider') ?? 'cricket_data_org';
    $api_key = $config->get('api_key');
    $cache_time = $config->get('cache_time') ?? 600;

    try {
      switch ($provider) {
        case 'cricket_data_org':
          $series_info = $this->fetchCricketDataOrgSeriesInfo($series_id);
          break;

        case 'api_cricket':
          $series_info = $this->fetchApiCricketSeriesInfo($series_id, $api_key);
          break;

        case 'api_sports':
          $series_info = $this->fetchApiSportsSeriesInfo($series_id, $api_key);
          break;

        default:
          $series_info = [];
      }

      $this->cache->set($cache_id, $series_info, time() + $cache_time);
      return $series_info;
    }
    catch (\Exception $e) {
      $this->logger->error('Error fetching series info: @message', ['@message' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * Get detailed information about a specific match.
   *
   * @param string $match_id
   *   The match ID to fetch information for.
   *
   * @return array
   *   Match details.
   */
  public function getMatchInfo($match_id) {
    $cache_id = 'cricket_chatbot:match_info:' . $match_id;

    if ($cache = $this->cache->get($cache_id)) {
      return $cache->data;
    }

    $config = $this->configFactory->get('cricket_chatbot.settings');
    $provider = $config->get('api_provider') ?? 'cricket_data_org';
    $api_key = $config->get('api_key');
    $cache_time = $config->get('cache_time') ?? 600;

    try {
      switch ($provider) {
        case 'cricket_data_org':
          $match_info = $this->fetchCricketDataOrgMatchInfo($match_id);
          break;

        case 'api_cricket':
          $match_info = $this->fetchApiCricketMatchInfo($match_id, $api_key);
          break;

        case 'api_sports':
          $match_info = $this->fetchApiSportsMatchInfo($match_id, $api_key);
          break;

        default:
          $match_info = [];
      }

      // Use shorter cache time for live matches
      $is_live = !empty($match_info['status']) &&
        (strpos(strtolower($match_info['status']), 'live') !== FALSE ||
         strpos(strtolower($match_info['status']), 'progress') !== FALSE);

      $cache_duration = $is_live ? min(300, $cache_time) : $cache_time;

      $this->cache->set($cache_id, $match_info, time() + $cache_duration);
      return $match_info;
    }
    catch (\Exception $e) {
      $this->logger->error('Error fetching match info: @message', ['@message' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * Get information about a specific player.
   *
   * @param string $player_name
   *   The player name to search for.
   *
   * @return array
   *   Player information.
   */
  public function getPlayerInfo($player_name) {
    $cache_id = 'cricket_chatbot:player_info:' . strtolower(str_replace(' ', '_', $player_name));

    if ($cache = $this->cache->get($cache_id)) {
      return $cache->data;
    }

    $config = $this->configFactory->get('cricket_chatbot.settings');
    $provider = $config->get('api_provider') ?? 'cricket_data_org';
    $api_key = $config->get('api_key');
    $cache_time = $config->get('cache_time') ?? 3600; // Longer cache for player info

    try {
      switch ($provider) {
        case 'cricket_data_org':
          $player_info = $this->fetchCricketDataOrgPlayerInfo($player_name);
          break;

        case 'api_cricket':
          $player_info = $this->fetchApiCricketPlayerInfo($player_name, $api_key);
          break;

        case 'api_sports':
          $player_info = $this->fetchApiSportsPlayerInfo($player_name, $api_key);
          break;

        default:
          $player_info = [];
      }

      $this->cache->set($cache_id, $player_info, time() + $cache_time);
      return $player_info;
    }
    catch (\Exception $e) {
      $this->logger->error('Error fetching player info: @message', ['@message' => $e->getMessage()]);
      return [];
    }
  }

  /**
   * Fetch matches from Cricket Data.org API.
   *
   * @param string $type
   *   The type of matches to fetch ('current', 'upcoming', or 'recent').
   *
   * @return array
   *   An array of matches.
   */
  private function fetchCricketDataOrgMatches($type) {
    $endpoint = 'https://api.cricapi.com/v1/';

    switch ($type) {
      case 'current':
        $endpoint .= 'currentMatches';
        break;

      case 'upcoming':
        $endpoint .= 'matches';
        break;

      case 'recent':
        $endpoint .= 'matches';
        break;

      default:
        throw new \InvalidArgumentException('Invalid match type');
    }
    $config = $this->configFactory->get('cricket_chatbot.settings');
    $api_key = $config->get('api_key');

    $options = [
      'query' => [
        'apikey' => $api_key, // Placeholder for demo
        'offset' => 0,
      ],
    ];

    if ($type === 'upcoming') {
      $options['query']['upcoming'] = true;
    }

    try {
      $response = $this->httpClient->request('GET', $endpoint, $options);
      $data = json_decode($response->getBody(), TRUE);

      // Process the API response into a standardized format
      $matches = [];

      if (isset($data['data']) && is_array($data['data'])) {
        foreach ($data['data'] as $match) {
          $matchData = [
            'id' => $match['id'] ?? '',
            'team1' => $match['teams'][0] ?? '',
            'team2' => $match['teams'][1] ?? '',
            'status' => $match['status'] ?? '',
          ];

          if ($type === 'current') {
            if (isset($match['score'][0]['r']) && isset($match['score'][1]['r'])) {
              $matchData['score'] = $match['teams'][0] . ': ' . $match['score'][0]['r'] . '/' .
                ($match['score'][0]['w'] ?? 0) . ' (' . ($match['score'][0]['o'] ?? 0) . ' overs), ' .
                $match['teams'][1] . ': ' . $match['score'][1]['r'] . '/' .
                ($match['score'][1]['w'] ?? 0) . ' (' . ($match['score'][1]['o'] ?? 0) . ' overs)';
            }
          }

          if ($type === 'upcoming' || $type === 'recent') {
            $matchData['date'] = $match['date'] ?? '';
            $matchData['venue'] = $match['venue'] ?? '';
          }

          if ($type === 'recent') {
            $matchData['result'] = $match['status'] ?? '';
          }

          $matches[] = $matchData;
        }
      }

      return $matches;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Error fetching matches from Cricket Data.org: @message', ['@message' => $e->getMessage()]);
      throw $e;
    }
  }

  /**
   * Fetch series info from Cricket Data.org API.
   *
   * @param string|null $series_id
   *   The series ID to fetch information for.
   *
   * @return array
   *   Series information.
   */
  private function fetchCricketDataOrgSeriesInfo($series_id = NULL) {
    $endpoint = 'https://api.cricapi.com/v1/series_info';
    $config = $this->configFactory->get('cricket_chatbot.settings');
    $api_key = $config->get('api_key');
    $options = [
      'query' => [
        'apikey' => $api_key, // Placeholder for demo
        'offset' => 0,
      ],
    ];

    if ($series_id) {
      $options['query']['id'] = $series_id;
    }

    try {
      $response = $this->httpClient->request('GET', $endpoint, $options);
      $data = json_decode($response->getBody(), TRUE);

      // Process the API response into a standardized format
      $series_info = [];

      if (isset($data['data']) && !empty($data['data'])) {
        $series = $data['data'];

        $series_info = [
          'id' => $series['id'] ?? '',
          'name' => $series['name'] ?? '',
          'startDate' => $series['startDate'] ?? '',
          'endDate' => $series['endDate'] ?? '',
          'matches' => [],
          'standings' => [],
        ];

        // Add matches if available
        if (isset($series['matches']) && is_array($series['matches'])) {
          foreach ($series['matches'] as $match) {
            $series_info['matches'][] = [
              'id' => $match['id'] ?? '',
              'name' => $match['name'] ?? '',
              'date' => $match['date'] ?? '',
              'venue' => $match['venue'] ?? '',
              'status' => $match['status'] ?? '',
            ];
          }
        }

        // Add standings if available
        if (isset($series['standings']) && is_array($series['standings'])) {
          foreach ($series['standings'] as $standing) {
            $series_info['standings'][] = [
              'team' => $standing['team'] ?? '',
              'position' => $standing['pos'] ?? '',
              'points' => $standing['points'] ?? '',
              'played' => $standing['played'] ?? '',
              'won' => $standing['won'] ?? '',
              'lost' => $standing['lost'] ?? '',
              'nrr' => $standing['nrr'] ?? '',
            ];
          }
        }
      }

      return $series_info;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Error fetching series info from Cricket Data.org: @message', ['@message' => $e->getMessage()]);
      throw $e;
    }
  }

  /**
   * Fetch match info from Cricket Data.org API.
   *
   * @param string $match_id
   *   The match ID to fetch information for.
   *
   * @return array
   *   Match details.
   */
  private function fetchCricketDataOrgMatchInfo($match_id) {
    $endpoint = 'https://api.cricapi.com/v1/match_info';
    $config = $this->configFactory->get('cricket_chatbot.settings');
    $api_key = $config->get('api_key');
    $options = [
      'query' => [
        'apikey' => $api_key, // Placeholder for demo
        'id' => $match_id,
      ],
    ];

    try {
      $response = $this->httpClient->request('GET', $endpoint, $options);
      $data = json_decode($response->getBody(), TRUE);

      // Process the API response into a standardized format
      $match_info = [];

      if (isset($data['data']) && !empty($data['data'])) {
        $match = $data['data'];

        $match_info = [
          'id' => $match['id'] ?? '',
          'name' => $match['name'] ?? '',
          'matchType' => $match['matchType'] ?? '',
          'status' => $match['status'] ?? '',
          'venue' => $match['venue'] ?? '',
          'date' => $match['date'] ?? '',
          'dateTimeGMT' => $match['dateTimeGMT'] ?? '',
          'teams' => $match['teams'] ?? [],
          'score' => [],
          'tossWinner' => $match['tossWinner'] ?? '',
          'tossChoice' => $match['tossChoice'] ?? '',
          'matchWinner' => $match['matchWinner'] ?? '',
        ];

        // Add scores if available
        if (isset($match['score']) && is_array($match['score'])) {
          foreach ($match['score'] as $score) {
            $match_info['score'][] = [
              'team' => $score['inning'] ?? '',
              'runs' => $score['r'] ?? 0,
              'wickets' => $score['w'] ?? 0,
              'overs' => $score['o'] ?? 0,
            ];
          }
        }

        // Add scorecard if available
        if (isset($match['scorecard']) && is_array($match['scorecard'])) {
          $match_info['scorecard'] = [];

          foreach ($match['scorecard'] as $inning) {
            $inning_data = [
              'inning' => $inning['inning'] ?? '',
              'batting' => [],
              'bowling' => [],
            ];

            // Add batting details
            if (isset($inning['batting']) && is_array($inning['batting'])) {
              foreach ($inning['batting'] as $batsman) {
                $inning_data['batting'][] = [
                  'name' => $batsman['name'] ?? '',
                  'runs' => $batsman['r'] ?? 0,
                  'balls' => $batsman['b'] ?? 0,
                  'fours' => $batsman['4s'] ?? 0,
                  'sixes' => $batsman['6s'] ?? 0,
                  'strikeRate' => $batsman['sr'] ?? 0,
                  'dismissal' => $batsman['dismissal-text'] ?? 'not out',
                ];
              }
            }

            // Add bowling details
            if (isset($inning['bowling']) && is_array($inning['bowling'])) {
              foreach ($inning['bowling'] as $bowler) {
                $inning_data['bowling'][] = [
                  'name' => $bowler['name'] ?? '',
                  'overs' => $bowler['o'] ?? 0,
                  'maidens' => $bowler['m'] ?? 0,
                  'runs' => $bowler['r'] ?? 0,
                  'wickets' => $bowler['w'] ?? 0,
                  'economy' => $bowler['eco'] ?? 0,
                ];
              }
            }

            $match_info['scorecard'][] = $inning_data;
          }
        }
      }

      return $match_info;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Error fetching match info from Cricket Data.org: @message', ['@message' => $e->getMessage()]);
      throw $e;
    }
  }

  /**
   * Fetch player info from Cricket Data.org API.
   *
   * @param string $player_name
   *   The player name to search for.
   *
   * @return array
   *   Player information.
   */
  private function fetchCricketDataOrgPlayerInfo($player_name) {
    $endpoint = 'https://api.cricapi.com/v1/players';
    $config = $this->configFactory->get('cricket_chatbot.settings');
    $api_key = $config->get('api_key');
    $options = [
      'query' => [
        'apikey' => $api_key, // Placeholder for demo
        'search' => $player_name,
      ],
    ];

    try {
      $response = $this->httpClient->request('GET', $endpoint, $options);
      $data = json_decode($response->getBody(), TRUE);

      // Process the API response into a standardized format
      $player_info = [];

      if (isset($data['data']) && is_array($data['data']) && !empty($data['data'])) {
        // Get the first matching player
        $player = $data['data'][0];

        $player_info = [
          'id' => $player['id'] ?? '',
          'name' => $player['name'] ?? '',
          'country' => $player['country'] ?? '',
          'role' => $player['role'] ?? '',
          'birthplace' => $player['birthplace'] ?? '',
          'battingStyle' => $player['battingStyle'] ?? '',
          'bowlingStyle' => $player['bowlingStyle'] ?? '',
        ];

        // If player ID is available, fetch detailed stats
        if (!empty($player_info['id'])) {
          $player_info = array_merge($player_info, $this->fetchCricketDataOrgPlayerStats($player_info['id']));
        }
      }

      return $player_info;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Error fetching player info from Cricket Data.org: @message', ['@message' => $e->getMessage()]);
      throw $e;
    }
  }

  /**
   * Fetch detailed player stats from Cricket Data.org API.
   *
   * @param string $player_id
   *   The player ID to fetch statistics for.
   *
   * @return array
   *   Player statistics.
   */
  private function fetchCricketDataOrgPlayerStats($player_id) {
    $endpoint = 'https://api.cricapi.com/v1/players_info';
    $config = $this->configFactory->get('cricket_chatbot.settings');
    $api_key = $config->get('api_key');
    $options = [
      'query' => [
        'apikey' => $api_key, // Placeholder for demo
        'id' => $player_id,
      ],
    ];

    try {
      $response = $this->httpClient->request('GET', $endpoint, $options);
      $data = json_decode($response->getBody(), TRUE);

      // Process the API response into a standardized format
      $player_stats = [
        'stats' => [
          'batting' => [],
          'bowling' => [],
        ],
      ];

      if (isset($data['data']) && !empty($data['data'])) {
        $info = $data['data'];

        // Add career stats if available
        if (isset($info['stats']) && is_array($info['stats'])) {
          // Process batting stats
          if (isset($info['stats']['batting']) && is_array($info['stats']['batting'])) {
            foreach ($info['stats']['batting'] as $format => $stats) {
              $player_stats['stats']['batting'][$format] = [
                'matches' => $stats['matches'] ?? 0,
                'innings' => $stats['innings'] ?? 0,
                'runs' => $stats['runs'] ?? 0,
                'highestScore' => $stats['highest'] ?? 0,
                'average' => $stats['average'] ?? 0,
                'strikeRate' => $stats['strike'] ?? 0,
                'centuries' => $stats['100s'] ?? 0,
                'halfCenturies' => $stats['50s'] ?? 0,
                'fours' => $stats['4s'] ?? 0,
                'sixes' => $stats['6s'] ?? 0,
              ];
            }
          }

          // Process bowling stats
          if (isset($info['stats']['bowling']) && is_array($info['stats']['bowling'])) {
            foreach ($info['stats']['bowling'] as $format => $stats) {
              $player_stats['stats']['bowling'][$format] = [
                'matches' => $stats['matches'] ?? 0,
                'innings' => $stats['innings'] ?? 0,
                'wickets' => $stats['wickets'] ?? 0,
                'bestInnings' => $stats['best_innings'] ?? '',
                'bestMatch' => $stats['best_match'] ?? '',
                'economy' => $stats['econ'] ?? 0,
                'average' => $stats['average'] ?? 0,
                'strikeRate' => $stats['strike'] ?? 0,
                'fiveWickets' => $stats['5w'] ?? 0,
                'tenWickets' => $stats['10w'] ?? 0,
              ];
            }
          }
        }
      }

      return $player_stats;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Error fetching player stats from Cricket Data.org: @message', ['@message' => $e->getMessage()]);
      return ['stats' => ['batting' => [], 'bowling' => []]];
    }
  }

  /**
   * Fetch matches from API-Cricket.
   *
   * @param string $type
   *   The type of matches to fetch ('current', 'upcoming', or 'recent').
   * @param string $api_key
   *   The API key.
   *
   * @return array
   *   An array of matches.
   */
  private function fetchApiCricketMatches($type, $api_key) {
    $endpoint = 'https://api-cricket.p.rapidapi.com/fixtures';

    $options = [
      'headers' => [
        'x-rapidapi-host' => 'api-cricket.p.rapidapi.com',
        'x-rapidapi-key' => $api_key,
      ],
      'query' => [
        'league' => '1',
      ],
    ];

    switch ($type) {
      case 'current':
        $options['query']['live'] = 'all';
        break;

      case 'upcoming':
        $options['query']['date'] = date('Y-m-d', strtotime('+1 day'));
        break;

      case 'recent':
        $options['query']['date'] = date('Y-m-d', strtotime('-1 day'));
        break;

      default:
        throw new \InvalidArgumentException('Invalid match type');
    }

    try {
      $response = $this->httpClient->request('GET', $endpoint, $options);
      $data = json_decode($response->getBody(), TRUE);

      // Process the API response into a standardized format
      $matches = [];

      if (isset($data['response']) && is_array($data['response'])) {
        foreach ($data['response'] as $match) {
          $matchData = [
            'id' => $match['fixture']['id'] ?? '',
            'team1' => $match['teams']['home']['name'] ?? '',
            'team2' => $match['teams']['away']['name'] ?? '',
            'status' => $match['fixture']['status']['long'] ?? '',
          ];

          if ($type === 'current') {
            if (isset($match['score'])) {
              $matchData['score'] = $match['teams']['home']['name'] . ': ' .
                ($match['scores']['home'] ?? '0') . ', ' .
                $match['teams']['away']['name'] . ': ' .
                ($match['scores']['away'] ?? '0');
            }
          }

          if ($type === 'upcoming' || $type === 'recent') {
            $matchData['date'] = date('Y-m-d H:i', strtotime($match['fixture']['date'] ?? ''));
            $matchData['venue'] = $match['fixture']['venue']['name'] ?? '';
          }

          if ($type === 'recent') {
            $matchData['result'] = $match['fixture']['status']['long'] ?? '';
          }

          $matches[] = $matchData;
        }
      }

      return $matches;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Error fetching matches from API-Cricket: @message', ['@message' => $e->getMessage()]);
      throw $e;
    }
  }

  /**
   * Fetch matches from API-Sports Cricket.
   *
   * @param string $type
   *   The type of matches to fetch ('current', 'upcoming', or 'recent').
   * @param string $api_key
   *   The API key.
   *
   * @return array
   *   An array of matches.
   */
  private function fetchApiSportsMatches($type, $api_key) {
    $endpoint = 'https://cricket-api.p.rapidapi.com/matches';

    $options = [
      'headers' => [
        'x-rapidapi-host' => 'cricket-api.p.rapidapi.com',
        'x-rapidapi-key' => $api_key,
      ],
    ];

    switch ($type) {
      case 'current':
        $options['query']['live'] = 'true';
        break;

      case 'upcoming':
        $options['query']['upcoming'] = 'true';
        break;

      case 'recent':
        $options['query']['recent'] = 'true';
        break;

      default:
        throw new \InvalidArgumentException('Invalid match type');
    }

    try {
      $response = $this->httpClient->request('GET', $endpoint, $options);
      $data = json_decode($response->getBody(), TRUE);

      // Process the API response into a standardized format
      $matches = [];

      if (isset($data['matches']) && is_array($data['matches'])) {
        foreach ($data['matches'] as $match) {
          $matchData = [
            'id' => $match['id'] ?? '',
            'team1' => $match['team1'] ?? '',
            'team2' => $match['team2'] ?? '',
            'status' => $match['status'] ?? '',
          ];

          if ($type === 'current') {
            if (isset($match['score1']) && isset($match['score2'])) {
              $matchData['score'] = $match['team1'] . ': ' .
                ($match['score1'] ?? '0') . ', ' .
                $match['team2'] . ': ' .
                ($match['score2'] ?? '0');
            }
          }

          if ($type === 'upcoming' || $type === 'recent') {
            $matchData['date'] = $match['date'] ?? '';
            $matchData['venue'] = $match['venue'] ?? '';
          }

          if ($type === 'recent') {
            $matchData['result'] = $match['result'] ?? '';
          }

          $matches[] = $matchData;
        }
      }

      return $matches;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Error fetching matches from API-Sports: @message', ['@message' => $e->getMessage()]);
      throw $e;
    }
  }

  /**
   * Fetch series info from API-Cricket.
   *
   * @param string|null $series_id
   *   The series ID to fetch information for.
   * @param string $api_key
   *   The API key.
   *
   * @return array
   *   Series information.
   */
  private function fetchApiCricketSeriesInfo($series_id, $api_key) {
    // Implementation for API-Cricket
    // This is a placeholder implementation
    return [
      'id' => $series_id ?? '1',
      'name' => 'API-Cricket Series',
      'startDate' => date('Y-m-d'),
      'endDate' => date('Y-m-d', strtotime('+1 month')),
      'matches' => [],
      'standings' => [],
    ];
  }

  /**
   * Fetch match info from API-Cricket.
   *
   * @param string $match_id
   *   The match ID to fetch information for.
   * @param string $api_key
   *   The API key.
   *
   * @return array
   *   Match details.
   */
  private function fetchApiCricketMatchInfo($match_id, $api_key) {
    // Implementation for API-Cricket
    // This is a placeholder implementation
    return [
      'id' => $match_id,
      'name' => 'API-Cricket Match',
      'matchType' => 'T20',
      'status' => 'In Progress',
      'venue' => 'API Stadium',
      'date' => date('Y-m-d'),
      'dateTimeGMT' => date('Y-m-d H:i:s'),
      'teams' => ['Team A', 'Team B'],
      'score' => [],
      'tossWinner' => 'Team A',
      'tossChoice' => 'bat',
      'matchWinner' => '',
    ];
  }

  /**
   * Fetch player info from API-Cricket.
   *
   * @param string $player_name
   *   The player name to search for.
   * @param string $api_key
   *   The API key.
   *
   * @return array
   *   Player information.
   */
  private function fetchApiCricketPlayerInfo($player_name, $api_key) {
    // Implementation for API-Cricket
    // This is a placeholder implementation
    return [
      'id' => '1',
      'name' => $player_name,
      'country' => 'International',
      'role' => 'Batsman',
      'birthplace' => 'Unknown',
      'battingStyle' => 'Right-handed',
      'bowlingStyle' => 'Right-arm medium',
      'stats' => [
        'batting' => [
          'ODI' => [
            'matches' => 100,
            'innings' => 95,
            'runs' => 4000,
            'highestScore' => 150,
            'average' => 45.5,
            'strikeRate' => 95,
            'centuries' => 10,
            'halfCenturies' => 25,
            'fours' => 300,
            'sixes' => 50,
          ],
        ],
        'bowling' => [],
      ],
    ];
  }

  /**
   * Fetch series info from API-Sports.
   *
   * @param string|null $series_id
   *   The series ID to fetch information for.
   * @param string $api_key
   *   The API key.
   *
   * @return array
   *   Series information.
   */
  private function fetchApiSportsSeriesInfo($series_id, $api_key) {
    // Implementation for API-Sports
    // This is a placeholder implementation
    return [
      'id' => $series_id ?? '2',
      'name' => 'API-Sports Series',
      'startDate' => date('Y-m-d'),
      'endDate' => date('Y-m-d', strtotime('+1 month')),
      'matches' => [],
      'standings' => [],
    ];
  }

  /**
   * Fetch match info from API-Sports.
   *
   * @param string $match_id
   *   The match ID to fetch information for.
   * @param string $api_key
   *   The API key.
   *
   * @return array
   *   Match details.
   */
  private function fetchApiSportsMatchInfo($match_id, $api_key) {
    // Implementation for API-Sports
    // This is a placeholder implementation
    return [
      'id' => $match_id,
      'name' => 'API-Sports Match',
      'matchType' => 'T20',
      'status' => 'In Progress',
      'venue' => 'API Stadium',
      'date' => date('Y-m-d'),
      'dateTimeGMT' => date('Y-m-d H:i:s'),
      'teams' => ['Team A', 'Team B'],
      'score' => [],
      'tossWinner' => 'Team A',
      'tossChoice' => 'bat',
      'matchWinner' => '',
    ];
  }

  /**
   * Fetch player info from API-Sports.
   *
   * @param string $player_name
   *   The player name to search for.
   * @param string $api_key
   *   The API key.
   *
   * @return array
   *   Player information.
   */
  private function fetchApiSportsPlayerInfo($player_name, $api_key) {
    // Implementation for API-Sports
    // This is a placeholder implementation
    return [
      'id' => '2',
      'name' => $player_name,
      'country' => 'International',
      'role' => 'Batsman',
      'birthplace' => 'Unknown',
      'battingStyle' => 'Right-handed',
      'bowlingStyle' => 'Right-arm medium',
      'stats' => [
        'batting' => [
          'ODI' => [
            'matches' => 100,
            'innings' => 95,
            'runs' => 4000,
            'highestScore' => 150,
            'average' => 45.5,
            'strikeRate' => 95,
            'centuries' => 10,
            'halfCenturies' => 25,
            'fours' => 300,
            'sixes' => 50,
          ],
        ],
        'bowling' => [],
      ],
    ];
  }

}
