<?php

declare(strict_types=1);

namespace Drupal\critical_css_ui\Asset;

use Drupal\Core\Asset\AssetCollectionRendererInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Render\Markup;

/**
 * Decorates the CSS collection renderer service, adds Critical CSS.
 *
 * @see \Drupal\Core\Asset\CssCollectionRenderer
 */
class CssCollectionRenderer implements AssetCollectionRendererInterface {

  /**
   * The decorated CSS collection renderer.
   *
   * @var \Drupal\Core\Asset\AssetCollectionRendererInterface
   */
  protected AssetCollectionRendererInterface $cssCollectionRenderer;

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Critical CSS provider.
   *
   * @var \Drupal\critical_css_ui\Asset\CriticalCssProviderInterface
   */
  protected CriticalCssProviderInterface $criticalCssProvider;

  /**
   * Constructs a CssCollectionRenderer.
   *
   * @param \Drupal\Core\Asset\AssetCollectionRendererInterface $css_collection_renderer
   *   The decorated CSS collection renderer.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory.
   * @param \Drupal\critical_css_ui\Asset\CriticalCssProviderInterface $critical_css_provider
   *   The Critical CSS provider.
   */
  public function __construct(
    AssetCollectionRendererInterface $css_collection_renderer,
    ConfigFactoryInterface $config_factory,
    CriticalCssProviderInterface $critical_css_provider
  ) {
    $this->cssCollectionRenderer = $css_collection_renderer;
    $this->configFactory = $config_factory;
    $this->criticalCssProvider = $critical_css_provider;
  }

  /**
   * {@inheritdoc}
   */
  public function render(array $css_assets): array {
    // Fixes empty CSS assets.
    if (empty($css_assets)) {
      return [];
    }

    $css_assets = $this->cssCollectionRenderer->render($css_assets);
    if (!$this->criticalCssProvider->isEnabled() || $this->criticalCssProvider->isAlreadyProcessed()) {
      return $css_assets;
    }

    $new_css_assets = [];
    // Add critical CSS asset if found.
    $criticalCssAsset = $this->getCriticalCssAsset();
    if ($criticalCssAsset) {
      $new_css_assets[] = $criticalCssAsset;
    }

    // If a critical CSS is found, make other CSS files asynchronous.
    $criticalCss = $this->criticalCssProvider->getCriticalCss();
    if ($criticalCss) {
      $asyncAssets = $this->makeAssetsAsync($css_assets);
      $new_css_assets = array_merge($new_css_assets, $asyncAssets);
    }
    else {
      $new_css_assets = array_merge($new_css_assets, $css_assets);
    }

    return $new_css_assets;
  }

  /**
   * Get critical CSS element.
   *
   * @return array|null
   *   Null if no critical CSS found. Otherwise, an array with the following
   *   items: '#type', '#tag', '#attributes', '#value'.
   */
  protected function getCriticalCssAsset(): ?array {
    $criticalCss = $this->criticalCssProvider->getCriticalCss();
    if (!$criticalCss) {
      return NULL;
    }

    return [
      '#type' => 'html_tag',
      '#tag' => 'style',
      '#attributes' => ['id' => 'critical-css'],
      '#value' => Markup::create($criticalCss),
    ];
  }

  /**
   * Make CSS assets load asynchronously.
   *
   * @param array $css_assets
   *   Array of CSS assets.
   *
   * @return array
   *   Array of async CSS assets.
   */
  protected function makeAssetsAsync(array $css_assets): array {
    $asyncAssets = [];
    foreach ($css_assets as $asset) {
      // Skip files with print media.
      if (isset($asset['#attributes']['media']) && $asset['#attributes']['media'] === 'print') {
        $asyncAssets[] = $asset;
      }
      else {
        // Add a stylesheet link with print media and an "onload" event.
        // @see https://www.filamentgroup.com/lab/load-css-simpler/
        $onLoadAsset = $asset;
        $onLoadAsset['#attributes']['media'] = 'print';
        $onLoadAsset['#attributes']['data-onload-media'] = 'all';
        $onLoadAsset['#attributes']['onload'] = 'this.onload=null;this.media=this.dataset.onloadMedia';
        $asyncAssets[] = $onLoadAsset;

        // Add fallback element for non-JS browsers.
        $noScriptAsset = $asset;
        $noScriptAsset['#noscript'] = TRUE;
        $asyncAssets[] = $noScriptAsset;
      }
    }

    return $asyncAssets;
  }

}

