<?php

declare(strict_types=1);

namespace Drupal\critical_css_ui\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Form\DeleteMultipleForm;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\critical_css_ui\CriticalCSSInterface;
use Drupal\critical_css_ui\CriticalCSSListBuilder;
use Drupal\critical_css_ui\Form\CriticalCSSForm;
use Drupal\views\EntityViewsData;

/**
 * Defines the critical css entity class.
 */
#[ContentEntityType(
  id: 'critical_css',
  label: new TranslatableMarkup('Critical CSS'),
  label_collection: new TranslatableMarkup('Critical CSS'),
  label_singular: new TranslatableMarkup('critical css'),
  label_plural: new TranslatableMarkup('critical css'),
  entity_keys: [
    'id' => 'id',
    'label' => 'target_context',
    'published' => 'status',
    'uuid' => 'uuid',
  ],
  handlers: [
    'list_builder' => CriticalCSSListBuilder::class,
    'views_data' => EntityViewsData::class,
    'form' => [
      'add' => CriticalCSSForm::class,
      'edit' => CriticalCSSForm::class,
      'node' => \Drupal\critical_css_ui\Form\CriticalCssContextForm::class,
      'node_type' => \Drupal\critical_css_ui\Form\CriticalCssContextForm::class,
      'delete' => ContentEntityDeleteForm::class,
      'delete-multiple-confirm' => DeleteMultipleForm::class,
    ],
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
  links: [
    'collection' => '/admin/config/development/performance/critical-css',
    'add-form' => '/admin/config/development/performance/critical-css/add',
    'canonical' => '/admin/config/development/performance/critical-css/{critical_css}',
    'edit-form' => '/admin/config/development/performance/critical-css/{critical_css}/edit',
    'delete-form' => '/admin/config/development/performance/critical-css/{critical_css}/delete',
    'delete-multiple-form' => '/admin/content/critical-css/delete-multiple',
  ],
  admin_permission: 'administer critical_css',
  base_table: 'critical_css',
  label_count: [
    'singular' => '@count critical css',
    'plural' => '@count critical css',
  ],
)]
class CriticalCSS extends ContentEntityBase implements CriticalCSSInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['target_context'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Target Context'))
      ->setDescription(new TranslatableMarkup('The target context for this critical CSS (e.g., route name, path pattern).'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->addConstraint('UniqueField')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['css'] = BaseFieldDefinition::create('string_long')
      ->setLabel(new TranslatableMarkup('CSS'))
      ->setDescription(new TranslatableMarkup('The critical CSS content.'))
      ->setRequired(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'basic_string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => -4,
        'settings' => [
          'rows' => 20,
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(new TranslatableMarkup('Status'))
      ->setDefaultValue(TRUE)
      ->setSetting('on_label', 'Enabled')
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => FALSE,
        ],
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'boolean',
        'label' => 'above',
        'weight' => 0,
        'settings' => [
          'format' => 'enabled-disabled',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(new TranslatableMarkup('Authored on'))
      ->setDescription(new TranslatableMarkup('The time that the critical css was created.'))
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(new TranslatableMarkup('Changed'))
      ->setDescription(new TranslatableMarkup('The time that the critical css was last edited.'));

    return $fields;
  }

}
