<?php

declare(strict_types=1);

namespace Drupal\critical_css_ui\Asset;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Routing\ResettableStackedRouteMatchInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Critical CSS Provider.
 *
 * Gets Critical CSS from entities based on current request context.
 */
class CriticalCssProvider implements CriticalCssProviderInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * Current Route Match.
   *
   * @var \Drupal\Core\Routing\ResettableStackedRouteMatchInterface
   */
  protected ResettableStackedRouteMatchInterface $currentRouteMatch;

  /**
   * Current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The route admin context to determine whether a route is an admin one.
   *
   * @var \Drupal\Core\Routing\AdminContext
   */
  protected AdminContext $adminContext;

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Flag set when this request has already been processed.
   *
   * @var bool
   */
  protected bool $isAlreadyProcessed = FALSE;

  /**
   * Critical CSS data to be inlined.
   *
   * @var string
   */
  protected string $criticalCss = '';

  /**
   * Constructs a CriticalCssProvider.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   Request stack.
   * @param \Drupal\Core\Routing\ResettableStackedRouteMatchInterface $current_route_match
   *   Current route match.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Current user.
   * @param \Drupal\Core\Routing\AdminContext $admin_context
   *   The route admin context service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    RequestStack $request_stack,
    ResettableStackedRouteMatchInterface $current_route_match,
    AccountProxyInterface $current_user,
    AdminContext $admin_context,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->request = $request_stack->getCurrentRequest();
    $this->currentRouteMatch = $current_route_match;
    $this->currentUser = $current_user;
    $this->adminContext = $admin_context;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public function getCriticalCss(): string {
    // Return previous result, if any.
    if ($this->isAlreadyProcessed) {
      return $this->criticalCss;
    }

    $this->isAlreadyProcessed = TRUE;

    if (!$this->isEnabled()) {
      return '';
    }

    // Get possible target contexts to match.
    $targetContexts = $this->getTargetContexts();

    // Try to find a matching Critical CSS entity.
    $storage = $this->entityTypeManager->getStorage('critical_css');
    foreach ($targetContexts as $targetContext) {
      $entities = $storage->loadByProperties([
        'target_context' => $targetContext,
        'status' => TRUE,
      ]);

      if (!empty($entities)) {
        $entity = reset($entities);
        if ($entity instanceof FieldableEntityInterface) {
          $css = $entity->get('css')->value;
          if (!empty($css)) {
            $this->criticalCss = trim($css);
            break;
          }
        }
      }
    }

    return $this->criticalCss;
  }

  /**
   * Get all possible target contexts to match.
   *
   * @return array
   *   Array of target contexts to try, in order of specificity.
   */
  protected function getTargetContexts(): array {
    $contexts = [];

    // Get entity-specific contexts.
    $entity = $this->getCurrentEntity();
    if ($entity) {
      $entityType = $entity->getEntityTypeId();
      $bundle = $entity->bundle();
      $id = $entity->id();

      // Only use node entities.
      if ($entityType === 'node') {
        // Try entity type and ID first (most specific).
        $contexts[] = $entityType . ':' . $id;
        // Try entity type and bundle (less specific).
        $contexts[] = $entityType . ':' . $bundle;
      }
    }

    // Add default fallback.
    $contexts[] = 'default';

    // Remove duplicates and empty values.
    $contexts = array_filter(array_unique($contexts));

    return $contexts;
  }

  /**
   * Get current entity from route.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface|null
   *   The current entity, or NULL if not found.
   */
  protected function getCurrentEntity() {
    $route = $this->currentRouteMatch->getRouteObject();
    if (!$route) {
      return NULL;
    }

    // Try to get entity from route parameters.
    $parameters = $route->getOption('parameters') ?: [];
    foreach ($parameters as $param_name => $param_info) {
      if (isset($param_info['type']) && strpos($param_info['type'], 'entity:') === 0) {
        $entity = $this->currentRouteMatch->getParameter($param_name);
        if ($entity) {
          return $entity;
        }
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(): bool {
    $route = $this->currentRouteMatch->getRouteObject();
    $canViewAdmin = $this->currentUser->hasPermission('view the administration theme');
    $isAjaxRequest = $this->request->isXmlHttpRequest();

    // Disable on admin routes and AJAX requests.
    if (($route && $this->adminContext->isAdminRoute($route) && $canViewAdmin) || $isAjaxRequest) {
      return FALSE;
    }

    // Check if module is enabled via config.
    $config = $this->configFactory->get('critical_css_ui.settings');
    return (bool) $config->get('enabled');
  }

  /**
   * {@inheritdoc}
   */
  public function isAlreadyProcessed(): bool {
    return $this->isAlreadyProcessed;
  }

  /**
   * {@inheritdoc}
   */
  public function reset(): void {
    $this->isAlreadyProcessed = FALSE;
    $this->criticalCss = '';
  }

}
