<?php

declare(strict_types=1);

namespace Drupal\critical_css_ui\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form controller for the critical css entity edit forms.
 */
final class CriticalCSSForm extends ContentEntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // Validate target_context uniqueness.
    $target_context = $form_state->getValue('target_context');
    if (!empty($target_context) && is_array($target_context) && isset($target_context[0]['value'])) {
      $context = trim($target_context[0]['value']);
      $entity = $this->entity;

      $storage = $this->entityTypeManager->getStorage('critical_css');
      $query = $storage->getQuery()
        ->accessCheck(FALSE)
        ->condition('target_context', $context);

      if (!$entity->isNew()) {
        $query->condition('id', $entity->id(), '<>');
      }

      $existing = $query->execute();

      if (!empty($existing)) {
        $form_state->setErrorByName('target_context', $this->t('A Critical CSS entry with the target context %context already exists.', [
          '%context' => $context,
        ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);

    $logger_args = [
      '%label' => $this->entity->label(),
      'link' => $this->entity->toLink($this->t('View'))->toString(),
    ];

    switch ($result) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('New critical css %label has been created.', [
          '%label' => $this->entity->label(),
        ]));
        $this->logger('critical_css_ui')->notice('New critical css %label has been created.', $logger_args);
        // Redirect to collection (list builder) after creating.
        $form_state->setRedirectUrl($this->entity->toUrl('collection'));
        break;

      case SAVED_UPDATED:
        $this->messenger()->addStatus($this->t('The critical css %label has been updated.', [
          '%label' => $this->entity->label(),
        ]));
        $this->logger('critical_css_ui')->notice('The critical css %label has been updated.', $logger_args);
        // Redirect to edit form after updating.
        $form_state->setRedirectUrl($this->entity->toUrl('edit-form'));
        break;

      default:
        throw new \LogicException('Could not save the entity.');
    }

    return $result;
  }

}
