<?php

declare(strict_types=1);

namespace Drupal\critical_css_ui\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for managing Critical CSS for different contexts (node or node type).
 */
final class CriticalCssContextForm extends ContentEntityForm {

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $currentRouteMatch;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $instance = parent::create($container);
    $instance->currentRouteMatch = $container->get('current_route_match');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    // Load or create entity based on node or node type context.
    if ($this->entity->isNew()) {
      $targetContext = NULL;

      // First, check if we have a node parameter (individual node route).
      $node = $this->currentRouteMatch->getParameter('node');
      if ($node) {
        // Use node:{id} for individual node context.
        $targetContext = 'node:' . $node->id();
      }
      else {
        // Otherwise, check for node_type parameter (node type route).
        $node_type = $this->currentRouteMatch->getParameter('node_type');
        if ($node_type) {
          // Use node:{bundle} format (e.g., node:article).
          $targetContext = 'node:' . $node_type->id();
        }
      }

      if ($targetContext) {
        // Try to load existing Critical CSS entity.
        $storage = $this->entityTypeManager->getStorage('critical_css');
        $entities = $storage->loadByProperties([
          'target_context' => $targetContext,
        ]);

        if (!empty($entities)) {
          $this->entity = reset($entities);
        }
        else {
          // Set default values for new entity.
          $this->entity->set('target_context', $targetContext);
          $this->entity->set('status', TRUE);
        }
      }
    }

    $form = parent::form($form, $form_state);

    // Hide target_context field as it's automatically set.
    if (isset($form['target_context'])) {
      $form['target_context']['#access'] = FALSE;
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);

    // Determine context for logging.
    $context_type = NULL;
    $context_id = NULL;
    $node = $this->currentRouteMatch->getParameter('node');
    if ($node) {
      $context_type = 'node';
      $context_id = $node->id();
    }
    else {
      $node_type = $this->currentRouteMatch->getParameter('node_type');
      if ($node_type) {
        $context_type = 'node_type';
        $context_id = $node_type->id();
      }
    }

    $logger_args = [
      '%label' => $this->entity->label(),
      'link' => $this->entity->toLink($this->t('View'))->toString(),
    ];
    if ($context_type && $context_id) {
      $logger_args['%context_type'] = $context_type;
      $logger_args['%context_id'] = $context_id;
    }

    switch ($result) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('New critical css %label has been created.', [
          '%label' => $this->entity->label(),
        ]));
        if ($context_type && $context_id) {
          $this->logger('critical_css_ui')->notice('New critical css %label has been created for %context_type %context_id.', $logger_args);
        }
        else {
          $this->logger('critical_css_ui')->notice('New critical css %label has been created.', $logger_args);
        }
        break;

      case SAVED_UPDATED:
        $this->messenger()->addStatus($this->t('The critical css %label has been updated.', [
          '%label' => $this->entity->label(),
        ]));
        if ($context_type && $context_id) {
          $this->logger('critical_css_ui')->notice('The critical css %label has been updated for %context_type %context_id.', $logger_args);
        }
        else {
          $this->logger('critical_css_ui')->notice('The critical css %label has been updated.', $logger_args);
        }
        break;

      default:
        throw new \LogicException('Could not save the entity.');
    }

    // Redirect after save.
    $node_type = $this->currentRouteMatch->getParameter('node_type');
    if ($node_type) {
      $form_state->setRedirectUrl($node_type->toUrl('edit-form'));
    }
    else {
      // If accessed from node, redirect to node.
      $node = $this->currentRouteMatch->getParameter('node');
      if ($node) {
        $form_state->setRedirectUrl($node->toUrl('canonical'));
      }
    }

    return $result;
  }

}
