<?php

declare(strict_types=1);

namespace Drupal\critical_css_ui\Form;

use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Critical CSS UI settings.
 */
final class CriticalCssConfigForm extends ConfigFormBase {

  /**
   * The cache tags invalidator.
   *
   * @var \Drupal\Core\Cache\CacheTagsInvalidatorInterface
   */
  protected CacheTagsInvalidatorInterface $cacheTagsInvalidator;

  /**
   * Constructs a new CriticalCssConfigForm object.
   *
   * @param \Drupal\Core\Cache\CacheTagsInvalidatorInterface $cache_tags_invalidator
   *   The cache tags invalidator.
   */
  public function __construct(CacheTagsInvalidatorInterface $cache_tags_invalidator) {
    $this->cacheTagsInvalidator = $cache_tags_invalidator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('cache_tags.invalidator')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['critical_css_ui.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'critical_css_ui_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('critical_css_ui.settings');

    $form['enabled'] = [
      '#type'          => 'checkbox',
      '#title'         => $this->t('Enable Critical CSS'),
      '#description'   => $this->t('When enabled, Critical CSS will be injected into pages based on the configured target contexts.'),
      '#default_value' => $config->get('enabled'),
    ];

    $form['default_enabled'] = [
      '#type'          => 'checkbox',
      '#title'         => $this->t('Enable Default Critical CSS Fallback'),
      '#description'   => $this->t('If no context-specific Critical CSS is found, use the default CSS defined below.'),
      '#default_value' => $config->get('default_enabled'),
    ];

    $form['default_critical_css'] = [
      '#type'          => 'textarea',
      '#title'         => $this->t('Default Critical CSS'),
      '#description'   => $this->t('Enter the CSS to be used as a fallback for all pages.'),
      '#default_value' => $config->get('default_critical_css'),
      '#rows'          => 15,
      '#states'        => [
        'visible' => [
          ':input[name="default_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('critical_css_ui.settings')
      ->set('enabled', (bool) $form_state->getValue('enabled'))
      ->set('default_enabled', (bool) $form_state->getValue('default_enabled'))
      ->set('default_critical_css', $form_state->getValue('default_critical_css'))
      ->save();

    // Invalidate cache tags so changes take effect immediately.
    // Using library cache tag as a broad invalidation since CSS affects render.
    $this->cacheTagsInvalidator->invalidateTags(['library_info']);

    parent::submitForm($form, $form_state);
  }

}
