<?php

declare(strict_types=1);

namespace Drupal\Tests\critical_css_ui\Functional;

use Drupal\Component\Utility\Html;
use Drupal\critical_css_ui\Entity\CriticalCSS;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests Critical CSS UI functionality on HTML HEAD.
 *
 * @group critical_css_ui
 */
final class CriticalCssUiHeadTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['node', 'critical_css_ui'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->config('critical_css_ui.settings')->set('enabled', TRUE)->save();
  }

  /**
   * Test critical CSS inside <head> element.
   */
  public function testCriticalCssInHead(): void {
    // No critical CSS should be found when module is disabled.
    $this->config('critical_css_ui.settings')->set('enabled', FALSE)->save();
    $this->drupalGet('<front>');
    $this->assertSession()->responseNotContains('<style id="critical-css">');

    // Critical CSS should be found when module is enabled and entity exists.
    $this->config('critical_css_ui.settings')->set('enabled', TRUE)->save();

    // Create a default critical CSS entity.
    CriticalCSS::create([
      'target_context' => 'default',
      'css'             => '/* default-critical.css */',
      'status'          => TRUE,
    ])->save();

    \Drupal::service('critical_css_ui.provider')->reset();
    \drupal_flush_all_caches();
    $this->drupalGet('<front>');
    $this->assertSession()->responseContains('<style id="critical-css">/* default-critical.css */</style>');

    // Create a node type and node.
    NodeType::create(['type' => 'article'])->save();
    $node = Node::create([
      'type'  => 'article',
      'title' => 'Test Article',
      'status' => 1,
    ]);
    $node->save();

    // Create critical CSS for the node bundle.
    CriticalCSS::create([
      'target_context' => 'node:article',
      'css'             => '/* node-article.css */',
      'status'          => TRUE,
    ])->save();

    // Node bundle CSS should be found when accessing a node of that type.
    \Drupal::service('critical_css_ui.provider')->reset();
    drupal_flush_all_caches();
    $this->drupalGet('/node/' . $node->id());
    $this->assertSession()->responseContains('<style id="critical-css">/* node-article.css */</style>');

    // Create critical CSS for specific node ID.
    CriticalCSS::create([
      'target_context' => 'node:' . $node->id(),
      'css'             => '/* node-' . $node->id() . '.css */',
      'status'          => TRUE,
    ])->save();

    // Specific node CSS should be found when accessing that node.
    \Drupal::service('critical_css_ui.provider')->reset();
    drupal_flush_all_caches();
    $this->drupalGet('/node/' . $node->id());
    $this->assertSession()->responseContains('<style id="critical-css">/* node-' . $node->id() . '.css */</style>');

    // Disabled entity should not be used.
    // First, delete the enabled entity for this specific node to test fallback.
    $storage = \Drupal::entityTypeManager()->getStorage('critical_css');
    $enabled_entities = $storage->loadByProperties([
      'target_context' => 'node:' . $node->id(),
      'status' => TRUE,
    ]);
    if (!empty($enabled_entities)) {
      $enabled_entity = reset($enabled_entities);
      $enabled_entity->delete();
    }

    // Create a disabled entity with the same target_context.
    $disabled_entity = CriticalCSS::create([
      'target_context' => 'node:' . $node->id(),
      'css'             => '/* disabled css */',
      'status'          => FALSE,
    ]);
    $disabled_entity->save();

    \Drupal::service('critical_css_ui.provider')->reset();
    drupal_flush_all_caches();
    $this->drupalGet('/node/' . $node->id());
    $this->assertSession()->responseNotContains('/* disabled css */');
    // Should fall back to bundle CSS.
    $this->assertSession()->responseContains('<style id="critical-css">/* node-article.css */</style>');

    // No critical CSS should be found when user is logged in on admin routes.
    $this->drupalLogin($this->rootUser);
    $this->drupalGet('/admin');
    $this->assertSession()->responseNotContains('<style id="critical-css">');
  }

  /**
   * Test context matching priority.
   */
  public function testContextMatchingPriority(): void {
    // Create entities with different contexts.
    CriticalCSS::create([
      'target_context' => 'default',
      'css'             => '/* default */',
      'status'          => TRUE,
    ])->save();

    NodeType::create(['type' => 'article'])->save();
    $node = Node::create([
      'type'   => 'article',
      'title'  => 'Test Article',
      'status' => 1,
    ]);
    $node->save();

    CriticalCSS::create([
      'target_context' => 'node:article',
      'css'             => '/* bundle */',
      'status'          => TRUE,
    ])->save();

    CriticalCSS::create([
      'target_context' => 'node:' . $node->id(),
      'css'             => '/* specific */',
      'status'          => TRUE,
    ])->save();

    // Most specific (node ID) should be used.
    \Drupal::service('critical_css_ui.provider')->reset();
    drupal_flush_all_caches();
    $this->drupalGet('/node/' . $node->id());
    $this->assertSession()->responseContains('<style id="critical-css">/* specific */</style>');
    $this->assertSession()->responseNotContains('/* bundle */');
    $this->assertSession()->responseNotContains('/* default */');

    // Delete specific entity, should fall back to bundle.
    $entities = \Drupal::entityTypeManager()
      ->getStorage('critical_css')
      ->loadByProperties(['target_context' => 'node:' . $node->id()]);
    if (!empty($entities)) {
      $entity = reset($entities);
      $entity->delete();
    }

    \Drupal::service('critical_css_ui.provider')->reset();
    drupal_flush_all_caches();
    $this->drupalGet('/node/' . $node->id());
    $this->assertSession()->responseContains('<style id="critical-css">/* bundle */</style>');
    $this->assertSession()->responseNotContains('/* default */');
  }

  /**
   * Test that non-critical CSS is loaded asynchronously.
   */
  public function testAsyncCssLoading(): void {
    // Create default critical CSS.
    CriticalCSS::create([
      'target_context' => 'default',
      'css'             => '/* critical */',
      'status'          => TRUE,
    ])->save();

    \Drupal::service('critical_css_ui.provider')->reset();
    drupal_flush_all_caches();
    $html = $this->drupalGet('<front>');

    // Check that critical CSS is present.
    $this->assertSession()->responseContains('<style id="critical-css">/* critical */</style>');

    // Check that non-critical CSS has async loading attributes.
    $document = Html::load($html);
    $xpath = new \DOMXPath($document);

    // Find stylesheet links that should be async.
    $stylesheet_links = $xpath->query('//link[@rel="stylesheet" and @media="print" and @data-onload-media="all"]');
    $this->assertGreaterThan(0, $stylesheet_links->length, 'Non-critical CSS should have async loading attributes.');

    // Check for noscript fallback.
    $noscript_links = $xpath->query('//noscript/link[@rel="stylesheet"]');
    $this->assertGreaterThan(0, $noscript_links->length, 'Noscript fallback should be present.');
  }

  /**
   * Test that only node entities are matched.
   */
  public function testOnlyNodeEntitiesMatched(): void {
    // Create a default critical CSS.
    CriticalCSS::create([
      'target_context' => 'default',
      'css'             => '/* default */',
      'status'          => TRUE,
    ])->save();

    // Access a non-node page (front page).
    \Drupal::service('critical_css_ui.provider')->reset();
    drupal_flush_all_caches();
    $this->drupalGet('<front>');

    // Should use default since no node entity is present.
    $this->assertSession()->responseContains('<style id="critical-css">/* default */</style>');
  }

}
