<?php

declare(strict_types=1);

namespace Drupal\Tests\critical_css_ui\Unit;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Routing\ResettableStackedRouteMatchInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\critical_css_ui\Asset\CriticalCssProvider;
use Drupal\critical_css_ui\Entity\CriticalCSS;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Unit tests for CriticalCssProvider.
 *
 * @coversDefaultClass \Drupal\critical_css_ui\Asset\CriticalCssProvider
 * @group critical_css_ui
 */
final class CriticalCssProviderTest extends UnitTestCase {

  /**
   * The class under test.
   *
   * @var \Drupal\critical_css_ui\Asset\CriticalCssProvider
   */
  protected CriticalCssProvider $criticalCssProvider;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Mock config factory.
    $config_factory = $this->createMock(ConfigFactoryInterface::class);
    $config = $this->getConfigFactoryStub([
      'critical_css_ui.settings' => [
        'enabled' => TRUE,
      ],
    ]);
    $config_factory->method('get')->willReturnCallback(function ($name) use ($config) {
      return $config->get($name);
    });

    // Mock entity storage.
    $this->entityStorage = $this->createMock(EntityStorageInterface::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->entityTypeManager->method('getStorage')
      ->with('critical_css')
      ->willReturn($this->entityStorage);

    // Mock request stack.
    $request = $this->createMock(Request::class);
    $request->method('getPathInfo')->willReturn('/node/1');
    $request->method('isXmlHttpRequest')->willReturn(FALSE);
    $request_stack = $this->createMock(RequestStack::class);
    $request_stack->method('getCurrentRequest')->willReturn($request);

    // Mock route match.
    $route_match = $this->createMock(ResettableStackedRouteMatchInterface::class);
    $route_match->method('getRouteName')->willReturn('entity.node.canonical');
    $route_match->method('getRouteObject')->willReturn(NULL);
    $route_match->method('getParameter')->willReturn(NULL);

    // Mock current user.
    $current_user = $this->createMock(AccountProxyInterface::class);
    $current_user->method('isAnonymous')->willReturn(TRUE);
    $current_user->method('hasPermission')->willReturn(FALSE);

    // Mock admin context.
    $admin_context = $this->createMock(AdminContext::class);
    $admin_context->method('isAdminRoute')->willReturn(FALSE);

    $this->criticalCssProvider = new CriticalCssProvider(
      $this->entityTypeManager,
      $request_stack,
      $route_match,
      $current_user,
      $admin_context,
      $config_factory
    );
  }

  /**
   * Test getCriticalCss returns empty when disabled.
   */
  public function testGetCriticalCssWhenDisabled(): void {
    $config_factory = $this->createMock(ConfigFactoryInterface::class);
    $config = $this->getConfigFactoryStub([
      'critical_css_ui.settings' => [
        'enabled' => FALSE,
      ],
    ]);
    $config_factory->method('get')->willReturnCallback(function ($name) use ($config) {
      return $config->get($name);
    });

    // Mock request stack with a request.
    $request = $this->createMock(Request::class);
    $request->method('getPathInfo')->willReturn('/');
    $request->method('isXmlHttpRequest')->willReturn(FALSE);
    $request_stack = $this->createMock(RequestStack::class);
    $request_stack->method('getCurrentRequest')->willReturn($request);

    $provider = new CriticalCssProvider(
      $this->entityTypeManager,
      $request_stack,
      $this->createMock(ResettableStackedRouteMatchInterface::class),
      $this->createMock(AccountProxyInterface::class),
      $this->createMock(AdminContext::class),
      $config_factory
    );

    $this->assertEquals('', $provider->getCriticalCss());
  }

  /**
   * Test getCriticalCss returns CSS from entity.
   */
  public function testGetCriticalCssFromEntity(): void {
    // Mock field item list.
    $field_item_list = $this->createMock(FieldItemListInterface::class);
    $field_item = $this->createMock(FieldItemInterface::class);
    $field_item->value = '/* test css */';
    $field_item_list->method('__get')->with('value')->willReturn('/* test css */');
    $field_item_list->method('getValue')->willReturn([['value' => '/* test css */']]);

    // Mock entity.
    $entity = $this->createMock(CriticalCSS::class);
    $entity->method('get')->with('css')->willReturn($field_item_list);
    $entity->method('getEntityTypeId')->willReturn('critical_css');

    // Mock storage to return entity.
    $this->entityStorage->method('loadByProperties')
      ->willReturn([$entity]);

    $css = $this->criticalCssProvider->getCriticalCss();
    $this->assertEquals('/* test css */', trim($css));
  }

  /**
   * Test getCriticalCss returns empty when no entity found.
   */
  public function testGetCriticalCssWhenNoEntityFound(): void {
    // Mock storage to return empty.
    $this->entityStorage->method('loadByProperties')
      ->willReturn([]);

    $css = $this->criticalCssProvider->getCriticalCss();
    $this->assertEquals('', $css);
  }

  /**
   * Test isEnabled returns false when disabled.
   */
  public function testIsEnabledWhenDisabled(): void {
    $config_factory = $this->createMock(ConfigFactoryInterface::class);
    $config = $this->getConfigFactoryStub([
      'critical_css_ui.settings' => [
        'enabled' => FALSE,
      ],
    ]);
    $config_factory->method('get')->willReturnCallback(function ($name) use ($config) {
      return $config->get($name);
    });

    // Mock request stack with a request.
    $request = $this->createMock(Request::class);
    $request->method('isXmlHttpRequest')->willReturn(FALSE);
    $request_stack = $this->createMock(RequestStack::class);
    $request_stack->method('getCurrentRequest')->willReturn($request);

    // Mock route match.
    $route_match = $this->createMock(ResettableStackedRouteMatchInterface::class);
    $route_match->method('getRouteObject')->willReturn(NULL);

    $provider = new CriticalCssProvider(
      $this->entityTypeManager,
      $request_stack,
      $route_match,
      $this->createMock(AccountProxyInterface::class),
      $this->createMock(AdminContext::class),
      $config_factory
    );

    $this->assertFalse($provider->isEnabled());
  }

  /**
   * Test isEnabled returns true when enabled.
   */
  public function testIsEnabledWhenEnabled(): void {
    $this->assertTrue($this->criticalCssProvider->isEnabled());
  }

  /**
   * Test reset clears cached CSS.
   */
  public function testReset(): void {
    // Mock field item list.
    $field_item_list = $this->createMock(FieldItemListInterface::class);
    $field_item_list->method('__get')->with('value')->willReturn('/* test css */');

    // Mock entity.
    $entity = $this->createMock(CriticalCSS::class);
    $entity->method('get')->with('css')->willReturn($field_item_list);
    $entity->method('getEntityTypeId')->willReturn('critical_css');

    $this->entityStorage->method('loadByProperties')
      ->willReturn([$entity]);

    // Get CSS first time.
    $css1 = $this->criticalCssProvider->getCriticalCss();
    $this->assertEquals('/* test css */', trim($css1));

    // Reset and get again - should recalculate.
    $this->criticalCssProvider->reset();
    $css2 = $this->criticalCssProvider->getCriticalCss();
    $this->assertEquals('/* test css */', trim($css2));
  }

}
