<?php

namespace Drupal\critique_and_review\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a Template Form for Critique And Review.
 *
 * Each field is predefined to provide the structure of the review.
 */
class TemplateCritiqueAndReviewForm extends FormBase {

  /**
   * The current active database's master connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  private $database;
  /**
   * Node id.
   *
   * @var int
   */
  private $nid;

  /**
   * User id.
   *
   * @var int
   */
  private $uid;

  /**
   * Node version id.
   *
   * @var int
   */
  private $vid;

  private $userIsAdmin = FALSE;



  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $current_user = \Drupal::currentUser();
    if (in_array('administrator', $current_user->getRoles())) {
      $this->userIsAdmin = TRUE;
    }

    $this->database = \Drupal::database();
    $this->uid = \Drupal::currentUser()->id();
    if ($node = \Drupal::routeMatch()->getParameter('node')) {
      $this->nid = $node->nid->value;
      $this->vid = $node->vid->value;
    }
    else {
      $this->nid = NULL;
      $this->vid = NULL;
    }

    $config = $this->config('critique_and_review.settings');



    $form['description'] = [
      '#type' => 'item',
      '#markup' => $config->get('intro_text'),
    ];

    // Get template items from the config for the module, these are title
    // and description.
    $templateItems = $config->get('review_items_titles');
    $template_items_instructions = $config->get('review_items_instructions');
   // dpm($templateItems);
    //dpm($template_items_instructions);
    // Are there any review already
    // Get review items from the database. These are title
    // and value (actual review).
    $review_finished = FALSE;
    if($reviews = $this->get()){
   
      foreach ($reviews as $review) {
        if ($review->rev_finished == 1){
           $review_finished = TRUE;
        }
        //dsm($review);
       // dsm($review->rev_finished);
          $form[] = $this->printFormItem($review->rev_item_id, $review->review_title, $template_items_instructions, $review->review, $review->rev_finished);
      }
    }
    // There are no reviews yet, so print an empty form
    else {
      // Index, number of items to display on the form.
      $counter = 1;
      // Loop through the config review items and build form for these items
      // If value exists for the item show it.
      foreach ($templateItems as $item) {
        $form[] = $this->printFormItem($counter, $item, $template_items_instructions, $reviews, 0);
        $counter++;
      }
    }

    if(!$review_finished && $config->get('add_more_reviews')) {

        $form['container_add_more'] = [
          '#type' => 'details',
          '#title' => "Add more",
        ];

        $form['container_add_more']['item_add_more_title'] = [
          '#type' => 'textfield',
          '#title' => 'Item title',
          '#default_value'  => "",
          '#description'  => "Add your own template item",
        ];

        $form['container_add_more']['nicey_template_add_more_text'] = [
          '#type' => 'text_format',
          '#title' => "",
          '#format' => 'basic_html',
        ];

        $form['container_add_more']['helptext'] = [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#value' => $this->t('To add even more items submit the form and a new item will appear.'),
          '#attributes' => ['class' => ["review-module-template-helptext"]],
        ];
    }
   

    // Add a submit button that handles the submission of the form.
    if (($this->nid && !$review_finished) or ($this->nid && $this->userIsAdmin)) {

      $form['container_finalise']['nicey_template_finalise_review'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Finalise review'),
          '#description' => $this->t('Once you finalise your review you will not be able to edit or add review items to it any more! The Review will be finalised and available to the Content Author.'),
        ];

      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Save Draft'),
        '#attributes' => ['class' => ["review-module-template-form-submit"]],
      ];


    }
    
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  private function printFormItem($counter, $item, $template_items_instructions, $reviews, $rev_finished) {

    $form['container' . $counter] = [
      '#type' => 'details',
      '#title' => $item,
    ];

    $form['item_title' . $counter] = [
      '#type' => 'hidden',
      '#title' => 'Item title',
      '#value'  => $item,
    ];
    $form['item_id' . $counter] = [
      '#type' => 'hidden',
      '#title' => 'Item id',
      '#value'  => $counter,
    ];

    // check if array, If we are passing in the array it means we are passing in all reviews.
    if (is_array($reviews)) {
      // If item already has a review get it and display it.
      $body = $this->setInitialBody($item, $reviews);
    }
    // Passing in string, so use it as review body.
    else {
      $body = $reviews;
    }

    $title_instructions = '';
    if ($template_items_instructions) {
      if(isset($template_items_instructions[$counter])){
        $title_instructions = $template_items_instructions[$counter];
      }
    }

    $form['container' . $counter]['nicey_template_text' . $counter] = [
      '#type' => 'text_format',
      '#title' => $title_instructions,
      '#format' => 'basic_html',
      '#default_value'  => $body,
    ];

    // $form['container' . $counter]['word_count_info' . $counter] = [
    //   '#type' => 'textfield',
    //   '#title' => $this->t('Wordcount'),
    //   '#default_value'  => "0",
    //   '#readonly' => 1,
    //   '#attributes' => [
    //       'id' => 'rev_item_wordcount-' . $counter,
    //       'readonly' => 'readonly', // Make the input read-only
    //    ],
    //     '#size' => 5, // controls the visible width, // Limit input to 10 characters
    // ];

    // if finalised make them disabled
    if (($rev_finished == 1) && (!$this->userIsAdmin)) {
      $form['container' . $counter]['nicey_template_text' . $counter]['#disabled'] = TRUE;
    }
    else {
      $form['container' . $counter]['nicey_template_delete_item' . $counter] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Delete this Review Item'),
      '#description' => $this->t('The Review Item will be removed when you save, this action cannot be undone!'),
    ];

    }


    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'review_template_review_form';
  }


  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {

    $values = $form_state->cleanValues()->getValues();
    // If add more not empty.
    if (isset($values['item_add_more_title'])) {
      // check for duplicate titles
      // get all titles and compare
      foreach ($values as $key => $value) {

        if (str_starts_with($key, "nicey_template_text")) {

            $arr = explode('nicey_template_text', $key);
            $index = $arr[1];
            $item_title = $values['item_title' . $index];
            if($item_title === $values['item_add_more_title']){
              $form_state->setErrorByName('error_duplicat_titles', $this->t('Duplicate title, you have used the title: '.$item_title .' twice, please use a different title.'));
            }
        }
      }
    }
  }
    

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    $values = $form_state->cleanValues()->getValues();

    foreach ($values as $key => $value) {

      // if delete item checkbox is checked
      if (str_starts_with($key, "nicey_template_delete_item")) {
        if ($value == 1){
          // get the id of the review item from the  key, key is in the form nicey_template_delete_item15
          $parts = explode('nicey_template_delete_item', $key);
          $this->delete($parts[1]);
        }
      }

      if (str_starts_with($key, "nicey_template_text")) {

        $arr = explode('nicey_template_text', $key);
        $index = $arr[1];
        $item_title = $values['item_title' . $index];
        $this->set($item_title, $value["value"],$index,  $values['nicey_template_finalise_review']);
      }
    }

    // If add more field is not empty add it.
    if (isset($values['item_add_more_title']) && $values['item_add_more_title']) {
      $nextIndex = $this->getNextIndexValue();
     $this->set($values['item_add_more_title'], $values['nicey_template_add_more_text']['value'], $nextIndex,  $values['nicey_template_finalise_review']);
    }
  }

  /**
   * Write review items to the database.
   */
  private function set($review_title, $body, $rev_id, $rev_finished) {

    // If row already exists update, so get reviews to check
    $reviews = $this->get($rev_id);

    $rowExists = FALSE;

    foreach ($reviews as $review) {
      //  dpm( $reviews);
      $rowExists = TRUE;
    }

    // Already exists so update.
    if ($rowExists) {

      $this->database
        ->update('critique_and_review_reviews')
        ->fields([
          'review' => $body,
          'rev_finished' => $rev_finished,
        ])
        ->condition('uid', $this->uid)
        ->condition('nid', $this->nid)
        ->condition('vid', $this->vid)
        ->condition('review_title', $review_title)
        ->execute();
    }
    // doesn't exist so insert new record.
    else {
      $fields = [
        'uid' => $this->uid,
        'nid' => $this->nid,
        'vid' => $this->vid,
        'rev_item_id' => $rev_id,
        'rev_finished' => $rev_finished,
        'review_title' => $review_title,
        'review' => $body,
      ];
      $this->database
        ->insert('critique_and_review_reviews')
        ->fields($fields)
        ->execute();
    }
  }
  private function getNextIndexValue(){
    $query = $this->database->select('critique_and_review_reviews', 'u')
      ->fields('u')
      ->condition('u.uid', $this->uid)
      ->condition('u.nid', $this->nid)
      ->condition('u.vid', $this->vid);
    $result = $query->execute()->fetchAll();

    $nexIindex = 0;
    foreach($result as $res){
      if((int)$res->rev_item_id > $nexIindex){
        $nexIindex = (int)$res->rev_item_id;
      }
    }
    $nexIindex++;
    return $nexIindex;
  }

  /**
   * Read review items from the database.
   */
  private function get($rev_id = NULL) {

    $query = $this->database->select('critique_and_review_reviews', 'u')
      ->fields('u')
      ->condition('u.uid', $this->uid)
      ->condition('u.nid', $this->nid)
      ->condition('u.vid', $this->vid);

    if ($rev_id) {
      $query->condition('u.rev_item_id', $rev_id);
    }

    return $query->execute()->fetchAll();
  }

   /**
   * Delete Review Item.
   */
  private function delete($rev_id) {

    $this->database->delete('critique_and_review_reviews')
      ->condition('uid', $this->uid)
      ->condition('nid', $this->nid)
      ->condition('vid', $this->vid)
      ->condition('rev_item_id', $rev_id)
      ->execute();
  }

  /**
   * Find items.
   *
   * Loop through all items for this node, user, and revision,
   * they are all in the $reviews variable,
   * if item exists in the database, return it, else return null.
   */
  private function setInitialBody($text, $reviews) {

    if (!$reviews) {
      return NULL;
    }

    foreach ($reviews as $review) {
      if ($review->review_title === $text) {
        return $review->review;
      }
    }
    return NULL;
  }

}
