<?php

namespace Drupal\crm;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityHandlerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the access control handler for the contact entity type.
 */
class ContactAccessControlHandler extends EntityAccessControlHandler implements EntityHandlerInterface {

  /**
   * {@inheritdoc}
   */
  protected $viewLabelOperation = TRUE;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a ContactAccessControlHandler object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($entity_type);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    // Check admin permission first.
    if ($account->hasPermission($this->entityType->getAdminPermission())) {
      return AccessResult::allowed()->cachePerPermissions();
    }

    $bundle = $entity->bundle();

    switch ($operation) {
      case 'view label':
        $permissions = ['view any crm_contact label'];
        $permissions[] = "view any $bundle crm_contact label";

        return AccessResult::allowedIfHasPermissions($account, $permissions, 'OR');

      case 'view':
        $permissions = ['view any crm_contact'];
        $permissions[] = "view any $bundle crm_contact";

        $result = AccessResult::allowedIfHasPermissions($account, $permissions, 'OR');
        if ($result->isAllowed()) {
          return $result;
        }

        // Check for user-contact mapping.
        if ($this->hasUserContactMapping($entity, $account)) {
          return AccessResult::allowedIfHasPermission($account, 'view mapped crm_contact')
            ->addCacheContexts(['user'])
            ->addCacheTags(['crm_user_contact_mapping_list']);
        }

        return $result;

      case 'update':
        $permissions = ['edit any crm_contact'];
        $permissions[] = "edit any $bundle crm_contact";

        $result = AccessResult::allowedIfHasPermissions($account, $permissions, 'OR');
        if ($result->isAllowed()) {
          return $result;
        }

        // Check for user-contact mapping.
        if ($this->hasUserContactMapping($entity, $account)) {
          return AccessResult::allowedIfHasPermission($account, 'edit mapped crm_contact')
            ->addCacheContexts(['user'])
            ->addCacheTags(['crm_user_contact_mapping_list']);
        }

        return $result;

      case 'delete':
        $permissions = ['delete any crm_contact'];
        $permissions[] = "delete any $bundle crm_contact";

        return AccessResult::allowedIfHasPermissions($account, $permissions, 'OR');

      default:
        // No opinion.
        return AccessResult::neutral();

    }

  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    // Check admin permission first.
    if ($account->hasPermission($this->entityType->getAdminPermission())) {
      return AccessResult::allowed()->cachePerPermissions();
    }

    return AccessResult::allowedIfHasPermissions(
      $account,
      ['create crm_contact', "create $entity_bundle crm_contact"],
      'OR',
    );
  }

  /**
   * Checks if a user has a contact mapping to the given entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The contact entity.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return bool
   *   TRUE if a mapping exists, FALSE otherwise.
   */
  protected function hasUserContactMapping(EntityInterface $entity, AccountInterface $account): bool {
    $query = $this->entityTypeManager->getStorage('crm_user_contact_mapping')->getQuery();
    $query->condition('crm_contact', $entity->id());
    $query->condition('user', $account->id());
    $result = $query->accessCheck(FALSE)->execute();

    return !empty($result);
  }

}
