<?php

declare(strict_types=1);

namespace Drupal\crm\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\crm\CrmContactInterface;
use Drupal\crm\Relationship;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns responses for Crm routes.
 */
class RelationshipController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The relationship service.
   *
   * @var \Drupal\crm\Relationship
   */
  protected $relationshipService;

  /**
   * The controller constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\crm\Relationship $relationship_service
   *   The relationship service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, Relationship $relationship_service) {
    $this->entityTypeManager = $entity_type_manager;
    $this->relationshipService = $relationship_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('entity_type.manager'),
      $container->get('crm.relationship')
    );
  }

  /**
   * Builds the response.
   *
   * @param \Drupal\crm\CrmContactInterface $crm_contact
   *   The contact entity.
   *
   * @return array
   *   A render array for the relationship list.
   */
  public function build(CrmContactInterface $crm_contact): array {
    $contact_id = $crm_contact->id();

    $build['active'] = [
      '#type' => 'table',
      '#header' => $this->buildHeader(),
      '#title' => $this->t('Active Relationships'),
      '#rows' => $this->getActiveRelationships($contact_id),
      '#empty' => $this->t('There are no active relationships.'),
      '#cache' => [],
    ];

    $build['inactive'] = [
      '#type' => 'table',
      '#header' => $this->buildHeader(),
      '#title' => $this->t('Inactive Relationships'),
      '#rows' => $this->getInactiveRelationships($contact_id),
      '#empty' => $this->t('There are no inactive relationships.'),
      '#cache' => [],
    ];

    $build['#cache']['max-age'] = 0;

    // Add contextual links.
    $build['#contextual_links']['crm_contact_relationship'] = [
      'route_parameters' => ['crm_contact' => $contact_id],
    ];

    return $build;
  }

  /**
   * Builds the header.
   *
   * @return array
   *   An array of header cells.
   */
  protected function buildHeader() {
    $header['type'] = $this->t('Relationship');
    $header['contact'] = $this->t('Contact');
    $header['start_date'] = $this->t('Start Date');
    $header['end_date'] = $this->t('End Date');
    $header['address'] = $this->t('Address');
    $header['email'] = $this->t('Email');
    $header['phone'] = $this->t('Phone');

    $header['operations'] = $this->t('Operations');
    return $header;
  }

  /**
   * Gets the active relationships.
   *
   * @param int $crm_contact
   *   The CRM contact ID.
   *
   * @return array
   *   An array of active relationships.
   */
  protected function getActiveRelationships($crm_contact) {

    $storage = $this->entityTypeManager->getStorage('crm_relationship');
    $query = $storage->getQuery();
    $query->condition('status', 1)
      ->sort('start_date', 'DESC');
    $or = $query->orConditionGroup()
      ->condition('contacts', $crm_contact);
    $query->condition($or);
    $result = $query->accessCheck(FALSE)->execute();
    $relationships = $storage->loadMultiple($result);

    return $this->doRows($relationships, $crm_contact);
  }

  /**
   * Gets the inactive relationships.
   *
   * @param int $crm_contact
   *   The CRM contact ID.
   *
   * @return array
   *   An array of inactive relationships.
   */
  protected function getInactiveRelationships($crm_contact) {

    $storage = $this->entityTypeManager->getStorage('crm_relationship');
    $query = $storage->getQuery();
    $query->condition('status', 0)
      ->sort('end_date', 'DESC');
    $or = $query->orConditionGroup()
      ->condition('contacts', $crm_contact);
    $query->condition($or);
    $result = $query->accessCheck(FALSE)->execute();
    $relationships = $storage->loadMultiple($result);

    return $this->doRows($relationships, $crm_contact);
  }

  /**
   * Builds the rows.
   *
   * @param array $relationships
   *   An array of relationships.
   * @param int $crm_contact
   *   The CRM contact ID.
   *
   * @return array
   *   An array of rows.
   */
  protected function doRows($relationships, $crm_contact) {
    $rows = [];

    foreach ($relationships as $relationship) {
      $contacts = $relationship->get('contacts')->referencedEntities();
      $is_a = $contacts[0]->id() == $crm_contact;
      $contact = $is_a ? $contacts[1] : $contacts[0];
      $contact_label = $contact->toLink()->toString();
      $relationship_label = $is_a ? $relationship->bundle->entity->get('label_b') : $relationship->bundle->entity->get('label_a');
      $rows[] = [
        'type' => $relationship_label,
        'contact' => $contact_label,
        'start_date' => $relationship->get('start_date')->value,
        'end_date' => $relationship->get('end_date')->value,
        'address' => $contact->get('addresses')->primary()?->entity->label(),
        'email' => $contact->get('emails')->primary()?->entity->label(),
        'phone' => $contact->get('telephones')->primary()?->entity->label(),
        'operations' => [
          'data' => [
            '#type' => 'operations',
            '#links' => [
              'edit' => [
                'title' => $this->t('Edit'),
                'url' => $relationship->toUrl('edit-form'),
              ],
              'delete' => [
                'title' => $this->t('Delete'),
                'url' => $relationship->toUrl('delete-form'),
              ],
            ],
          ],
        ],
      ];
    }
    return $rows;
  }

  /**
   * Displays a page with eligible relationship types for a contact.
   *
   * @param \Drupal\crm\CrmContactInterface $crm_contact
   *   The contact entity.
   *
   * @return array
   *   A render array for the add relationship page.
   */
  public function addPage(CrmContactInterface $crm_contact): array {
    $eligible_types = $this->relationshipService->getEligibleRelationshipTypesForContact($crm_contact);

    if (empty($eligible_types)) {
      return [
        '#markup' => $this->t('There are no relationship types available for this contact.'),
      ];
    }

    $build = [
      '#theme' => 'entity_add_list',
      '#bundles' => [],
      '#add_bundle_message' => $this->t('There are no relationship types available for this contact.'),
      '#cache' => [
        'tags' => ['config:crm_relationship_type_list'],
      ],
    ];

    foreach ($eligible_types as $type_id => $info) {
      /** @var \Drupal\crm\CrmRelationshipTypeInterface $type */
      $type = $info['type'];
      $positions = $info['positions'];
      $is_asymmetric = (bool) $type->get('asymmetric');

      // For symmetric relationships, only show one entry (use first position).
      // For asymmetric relationships, show an entry for each eligible position.
      $positions_to_show = $is_asymmetric ? $positions : [reset($positions)];

      foreach ($positions_to_show as $position) {
        $label = $position === 'a' ? $type->get('label_b') : $type->get('label_a');
        if (empty($label)) {
          $label = $type->label();
        }

        // Use a unique key for each position to allow both A and B entries.
        $bundle_key = $is_asymmetric ? $type_id . '_' . $position : $type_id;

        // Use contact_a or contact_b query parameter based on position.
        $query_param = $position === 'a' ? 'contact_a' : 'contact_b';

        $build['#bundles'][$bundle_key] = [
          'label' => $label,
          'description' => $type->getDescription(),
          'add_link' => Link::createFromRoute($label, 'entity.crm_relationship.add_form', [
            'crm_relationship_type' => $type_id,
          ], [
            'query' => [
              $query_param => $crm_contact->id(),
            ],
          ]),
        ];
      }
    }

    return $build;
  }

}
