<?php

declare(strict_types=1);

namespace Drupal\crm\Drush\Commands;

use Drupal\crm\RelationshipStatisticsServiceInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * CRM Drush commands.
 */
class CrmCommands extends DrushCommands {

  /**
   * The relationship statistics service.
   *
   * @var \Drupal\crm\RelationshipStatisticsServiceInterface
   */
  protected RelationshipStatisticsServiceInterface $statisticsService;

  /**
   * Constructs a CrmCommands object.
   *
   * @param \Drupal\crm\RelationshipStatisticsServiceInterface $statistics_service
   *   The relationship statistics service.
   */
  public function __construct(RelationshipStatisticsServiceInterface $statistics_service) {
    parent::__construct();
    $this->statisticsService = $statistics_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('crm.relationship_statistics'),
    );
  }

  /**
   * Recalculates relationship statistics for all contacts.
   */
  #[CLI\Command(name: 'crm:recalculate-statistics', aliases: ['crm-rs'])]
  #[CLI\Option(name: 'batch-size', description: 'Number of contacts to process per batch.')]
  #[CLI\Usage(name: 'crm:recalculate-statistics', description: 'Recalculate all contact relationship statistics.')]
  #[CLI\Usage(name: 'crm:recalculate-statistics --batch-size=50', description: 'Recalculate with smaller batch size.')]
  public function recalculateStatistics(array $options = ['batch-size' => 100]): void {
    $batch_size = (int) $options['batch-size'];

    $this->logger()->notice('Starting relationship statistics recalculation...');

    $start_time = microtime(TRUE);
    $total_processed = $this->statisticsService->recalculateAll($batch_size);
    $elapsed_time = round(microtime(TRUE) - $start_time, 2);

    $this->logger()->success(dt('Recalculated statistics for @count contacts in @time seconds.', [
      '@count' => $total_processed,
      '@time' => $elapsed_time,
    ]));
  }

}
