<?php

declare(strict_types=1);

namespace Drupal\crm\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Entity\RevisionLogEntityTrait;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crm\ContactAccessControlHandler;
use Drupal\crm\ContactListBuilder;
use Drupal\crm\CrmContactInterface;
use Drupal\crm\Form\ContactForm;
use Drupal\inline_entity_form\Plugin\Field\FieldWidget\InlineEntityFormComplex;
use Drupal\crm\ContactViewsData;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\crm\Field\AgeFieldItemList;

/**
 * CRM contact.
 *
 * A contact can be a person, an organization, a household, etc.
 */
#[ContentEntityType(
  id: 'crm_contact',
  label: new TranslatableMarkup('CRM Contact'),
  label_collection: new TranslatableMarkup('CRM Contacts'),
  label_singular: new TranslatableMarkup('crm contact'),
  label_plural: new TranslatableMarkup('crm contacts'),
  label_count: [
    'singular' => '@count crm contact',
    'plural' => '@count crm contacts',
  ],
  bundle_label: new TranslatableMarkup('Contact type'),
  handlers: [
    'list_builder' => ContactListBuilder::class,
    'views_data' => ContactViewsData::class,
    'access' => ContactAccessControlHandler::class,
    'form' => [
      'default' => ContactForm::class,
      'delete' => ContentEntityDeleteForm::class,
    ],
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
  base_table: 'crm_contact',
  revision_table: 'crm_contact_revision',
  show_revision_ui: TRUE,
  bundle_entity_type: 'crm_contact_type',
  field_ui_base_route: 'entity.crm_contact_type.edit_form',
  translatable: FALSE,
  admin_permission: 'administer crm',
  entity_keys: [
    'id' => 'id',
    'revision' => 'revision_id',
    'bundle' => 'bundle',
    'label' => 'name',
    'uuid' => 'uuid',
    'status' => 'status',
  ],
  revision_metadata_keys: [
    'revision_user' => 'revision_uid',
    'revision_created' => 'revision_timestamp',
    'revision_log_message' => 'revision_log',
  ],
  links: [
    'add-page' => '/crm/contact/add',
    'add-form' => '/crm/contact/add/{crm_contact_type}',
    'canonical' => '/crm/contact/{crm_contact}',
    'edit-form' => '/crm/contact/{crm_contact}/edit',
    'delete-form' => '/crm/contact/{crm_contact}/delete',
    'collection' => '/admin/content/crm/contact',
  ],
)]
class Contact extends RevisionableContentEntityBase implements CrmContactInterface {
  use EntityChangedTrait;
  use RevisionLogEntityTrait;

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    // If type is person, set the label field to the name field.
    if ($this->hasField('full_name') && !$this->get('full_name')->isEmpty()) {
      $name_array = $this->get('full_name')->getValue();
      if (!empty($name_array)) {
        $name_array = $name_array[0];

        // Get field settings for full_name field.
        $field_definition = $this->getFieldDefinition('full_name');
        $preferred_field_reference = $field_definition->getSetting('preferred_field_reference');
        $preferred_field_reference_separator = $field_definition->getSetting('preferred_field_reference_separator') ?? ', ';
        $alternative_field_reference = $field_definition->getSetting('alternative_field_reference');
        $alternative_field_reference_separator = $field_definition->getSetting('alternative_field_reference_separator') ?? ', ';

        // Add preferred name if configured and field exists.
        if ($preferred_field_reference && $this->hasField($preferred_field_reference) && !$this->get($preferred_field_reference)->isEmpty()) {
          $preferred_values = array_map(function ($item) {
            return $item['value'];
          }, $this->get($preferred_field_reference)->getValue());
          $name_array['preferred'] = implode($preferred_field_reference_separator, $preferred_values);
        }

        // Add alternative names if configured and field exists.
        if ($alternative_field_reference && $this->hasField($alternative_field_reference) && !$this->get($alternative_field_reference)->isEmpty()) {
          $alternative_values = array_map(function ($alias) {
            return $alias['value'];
          }, $this->get($alternative_field_reference)->getValue());
          $name_array['alternative'] = implode($alternative_field_reference_separator, $alternative_values);
        }

        $name_format = $this->bundle->entity->getThirdPartySetting('crm', 'name_format') ?? 'default';
        $name_formatter = \Drupal::service('name.formatter');
        $formatted_name = $name_formatter->format($name_array, $name_format);
        $this->set('name', $formatted_name);
      }
    }

    if ($this->get('name')->isEmpty()) {
      $type = $this->get('bundle')->entity->label();
      $uuid = $this->uuid();
      $this->set('name', $type . ' ' . $uuid);
    }

    parent::preSave($storage);
    $this->setNewRevision();
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE) {
    parent::postSave($storage, $update);

    $detail_fields = ['emails', 'telephones', 'addresses'];
    foreach ($detail_fields as $detail_field) {
      if (!$this->hasField($detail_field)) {
        continue;
      }
      foreach ($this->get($detail_field) as $item) {
        $detail = $item->entity;
        if ($detail && $detail->get('crm_contact')->isEmpty()) {
          $detail->set('crm_contact', $this->id());
          $detail->save();
        }
      }

    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setRevisionable(TRUE)
      ->setLabel(t('Name'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['emails'] = BaseFieldDefinition::create('primary_entity_reference')
      ->setLabel(t('Emails'))
      ->setRevisionable(TRUE)
      ->setCardinality(-1)
      ->setSetting('target_type', 'crm_contact_detail')
      ->setSetting('handler_settings', ['target_bundles' => ['email']])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'primary_entity_reference_inline_form',
        'weight' => 8,
        'settings' => [
          'allow_new' => TRUE,
          'allow_existing' => FALSE,
          'removed_reference' => InlineEntityFormComplex::REMOVED_DELETE,
          'match_operator' => 'CONTAINS',
          'allow_duplicate' => FALSE,
          'form_mode' => 'default',
          'override_labels' => TRUE,
          'label_singular' => t('email'),
          'label_plural' => t('emails'),
          'collapsible' => FALSE,
          'collapsed' => FALSE,
          'revision' => TRUE,
        ],
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'entity_reference_entity_view',
        'weight' => -5,
        'settings' => [
          'view_mode' => 'default',
          'link' => FALSE,
        ],
      ]);

    $fields['telephones'] = BaseFieldDefinition::create('primary_entity_reference')
      ->setLabel(t('Telephones'))
      ->setRevisionable(TRUE)
      ->setCardinality(-1)
      ->setSetting('target_type', 'crm_contact_detail')
      ->setSetting('handler_settings', ['target_bundles' => ['telephone']])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'primary_entity_reference_inline_form',
        'weight' => 9,
        'settings' => [
          'allow_new' => TRUE,
          'allow_existing' => FALSE,
          'removed_reference' => InlineEntityFormComplex::REMOVED_DELETE,
          'match_operator' => 'CONTAINS',
          'allow_duplicate' => FALSE,
          'form_mode' => 'default',
          'override_labels' => TRUE,
          'label_singular' => t('telephone'),
          'label_plural' => t('telephones'),
          'collapsible' => FALSE,
          'collapsed' => FALSE,
          'revision' => TRUE,
        ],
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'entity_reference_entity_view',
        'weight' => -5,
        'settings' => [
          'view_mode' => 'default',
          'link' => FALSE,
        ],
      ]);

    $fields['addresses'] = BaseFieldDefinition::create('primary_entity_reference')
      ->setLabel('Addresses')
      ->setRevisionable(TRUE)
      ->setCardinality(-1)
      ->setSetting('target_type', 'crm_contact_detail')
      ->setSetting('handler_settings', ['target_bundles' => ['address']])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'primary_entity_reference_inline_form',
        'weight' => 10,
        'settings' => [
          'allow_new' => TRUE,
          'allow_existing' => FALSE,
          'removed_reference' => InlineEntityFormComplex::REMOVED_DELETE,
          'match_operator' => 'CONTAINS',
          'allow_duplicate' => FALSE,
          'form_mode' => 'default',
          'override_labels' => TRUE,
          'label_singular' => t('address'),
          'label_plural' => t('addresses'),
          'collapsible' => FALSE,
          'collapsed' => FALSE,
          'revision' => TRUE,
        ],
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'entity_reference_entity_view',
        'weight' => 0,
        'settings' => [
          'view_mode' => 'default',
          'link' => FALSE,
        ],
      ]);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Status'))
      ->setDescription(t('A boolean indicating whether the contact is active.'))
      ->setDefaultValue(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => TRUE,
        ],
        'weight' => 120,
      ])
      ->setSetting('on_label', 'Status')
      ->setDisplayConfigurable('form', TRUE);

    $fields['start_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Start Date'))
      ->setDescription(t('When the contact starts.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'datetime_type' => 'date',
      ])
      ->setDefaultValue('')
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRevisionable(TRUE);

    $fields['end_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('End Date'))
      ->setDescription(t('When the contact ends.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'datetime_type' => 'date',
      ])
      ->setDefaultValue('')
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRevisionable(TRUE);

    $fields['age'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Age'))
      ->setDescription(t('The age of the contact.'))
      ->setComputed(TRUE)
      ->setClass(AgeFieldItemList::class)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['relationship_statistics'] = BaseFieldDefinition::create('crm_relationship_statistics')
      ->setLabel(t('Relationships'))
      ->setDescription(t("Statistics about this contact's relationships."))
      ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'crm_relationship_statistics_default',
        'weight' => 15,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created on'))
      ->setDescription(t('The time that the contact was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the contact was last edited.'));

    return $fields;
  }

}
