<?php

declare(strict_types=1);

namespace Drupal\crm\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBundleBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crm\CrmContactDetailTypeInterface;
use Drupal\crm\ContactDetailTypeAccessControlHandler;
use Drupal\crm\ContactDetailTypeListBuilder;
use Drupal\crm\Form\ContactDetailTypeForm;

/**
 * Defines the CRM Contact Detail type configuration entity.
 */
#[ConfigEntityType(
  id: 'crm_contact_detail_type',
  label: new TranslatableMarkup('CRM Contact Detail type'),
  label_collection: new TranslatableMarkup('CRM Contact Detail types'),
  label_singular: new TranslatableMarkup('crm contact detail type'),
  label_plural: new TranslatableMarkup('crm contact details types'),
  label_count: [
    'singular' => '@count crm contact details type',
    'plural' => '@count crm contact details types',
  ],
  handlers: [
    'access' => ContactDetailTypeAccessControlHandler::class,
    'form' => [
      'add' => ContactDetailTypeForm::class,
      'edit' => ContactDetailTypeForm::class,
      'delete' => EntityDeleteForm::class,
    ],
    'list_builder' => ContactDetailTypeListBuilder::class,
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
  admin_permission: 'administer crm',
  bundle_of: 'crm_contact_detail',
  config_prefix: 'crm_contact_detail_type',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'uuid' => 'uuid',
  ],
  links: [
    'add-form' => '/admin/structure/crm/contact-detail-types/add',
    'edit-form' => '/admin/structure/crm/contact-detail-types/manage/{crm_contact_detail_type}',
    'delete-form' => '/admin/structure/crm/contact-detail-types/manage/{crm_contact_detail_type}/delete',
    'collection' => '/admin/structure/crm/contact-detail-types',
  ],
  config_export: [
    'id',
    'label',
    'description',
    'uuid',
  ],
)]
class ContactDetailType extends ConfigEntityBundleBase implements CrmContactDetailTypeInterface {

  /**
   * The machine name of this crm contact detail type.
   */
  protected string $id;

  /**
   * The human-readable name of the crm contact detail type.
   */
  protected string $label;

  /**
   * The description of the crm contact detail type.
   *
   * @var string|null
   */
  protected ?string $description;

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->description ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setDescription($description) {
    $this->description = $description;

    return $this;
  }

  /**
   * Is the detail type locked?
   */
  public function isLocked() {
    $locked = \Drupal::state()->get('crm.contact_detail_type.locked');
    return $locked[$this->id()] ?? FALSE;
  }

}
