<?php

namespace Drupal\crm\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBundleBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crm\RelationshipTypeListBuilder;
use Drupal\crm\Form\RelationshipTypeForm;

use Drupal\crm\CrmRelationshipTypeInterface;

/**
 * Defines the CRM Relationship type configuration entity.
 */
#[ConfigEntityType(
  id: 'crm_relationship_type',
  label: new TranslatableMarkup('CRM Relationship type'),
  label_collection: new TranslatableMarkup('CRM Relationship types'),
  label_singular: new TranslatableMarkup('crm relationship type'),
  label_plural: new TranslatableMarkup('crm relationships types'),
  label_count: [
    'singular' => '@count crm relationships type',
    'plural' => '@count crm relationships types',
  ],
  handlers: [
    'form' => [
      'add' => RelationshipTypeForm::class,
      'edit' => RelationshipTypeForm::class,
      'delete' => EntityDeleteForm::class,
    ],
    'list_builder' => RelationshipTypeListBuilder::class,
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
  admin_permission: 'administer crm',
  bundle_of: 'crm_relationship',
  config_prefix: 'crm_relationship_type',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'uuid' => 'uuid',
  ],
  links: [
    'add-form' => '/admin/structure/crm/relationship_types/add',
    'edit-form' => '/admin/structure/crm/relationship_types/manage/{crm_relationship_type}',
    'delete-form' => '/admin/structure/crm/relationship_types/manage/{crm_relationship_type}/delete',
    'collection' => '/admin/structure/crm/relationship_types',
  ],
  config_export: [
    'id',
    'label',
    'description',
    'asymmetric',
    'uuid',
    'contact_type_a',
    'contact_type_b',
    'label_a',
    'label_a_plural',
    'label_b',
    'label_b_plural',
    'limit_a',
    'limit_b',
    'limit_active_only',
    'valid_contacts_a',
    'valid_contacts_b',
    'readonly_contact_a',
    'readonly_contact_b',
  ],
)]
class RelationshipType extends ConfigEntityBundleBase implements CrmRelationshipTypeInterface {

  /**
   * The machine name of this crm relationship type.
   *
   * @var string
   */
  protected $id;

  /**
   * The human-readable name of the crm relationship type.
   *
   * @var string
   */
  protected $label;

  /**
   * The description of the crm relationship type.
   *
   * @var string
   */
  protected $description;

  /**
   * Is the bundle and label the same for both sides of the relationship?
   *
   * @var bool
   */
  protected $asymmetric;

  /**
   * The human-readable name of the crm relationship type.
   *
   * @var string
   */
  protected $label_a;

  /**
   * The plural form of label_a.
   *
   * @var string
   */
  protected $label_a_plural;

  /**
   * The human-readable name of the crm relationship type.
   *
   * @var string
   */
  protected $label_b;

  /**
   * The plural form of label_b.
   *
   * @var string
   */
  protected $label_b_plural;

  /**
   * The contact types for the first contact in the relationship.
   *
   * @var array
   */
  protected $contact_type_a;

  /**
   * The contact types for the second contact in the relationship.
   *
   * @var array
   */
  protected $contact_type_b;

  /**
   * Maximum number of relationships where a contact can be in position A.
   *
   * @var int|null
   */
  protected $limit_a;

  /**
   * Maximum number of relationships where a contact can be in position B.
   *
   * @var int|null
   */
  protected $limit_b;

  /**
   * Whether to only count active relationships toward the limit.
   *
   * @var bool
   */
  protected $limit_active_only = FALSE;

  /**
   * Valid contact IDs for the first contact in the relationship.
   *
   * @var array
   */
  protected $valid_contacts_a = [];

  /**
   * Valid contact IDs for the second contact in the relationship.
   *
   * @var array
   */
  protected $valid_contacts_b = [];

  /**
   * Whether Contact A is read-only after being set.
   *
   * @var bool
   */
  protected $readonly_contact_a = TRUE;

  /**
   * Whether Contact B is read-only after being set.
   *
   * @var bool
   */
  protected $readonly_contact_b = TRUE;

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->description ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setDescription($description) {
    $this->description = $description;

    return $this;
  }

  /**
   * Is the relationship type locked?
   */
  public function isLocked() {
    $locked = \Drupal::state()->get('crm.relationship_type.locked');
    return $locked[$this->id()] ?? FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    parent::calculateDependencies();

    if ($type_a = $this->get('contact_type_a')) {
      foreach ($type_a as $type) {
        $this->addDependency('config', 'crm.crm_contact_type.' . $type);
      }
    }
    if ($type_b = $this->get('contact_type_b')) {
      foreach ($type_b as $type) {
        $this->addDependency('config', 'crm.crm_contact_type.' . $type);
      }
    }

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getLimitA(): ?int {
    return $this->limit_a;
  }

  /**
   * {@inheritdoc}
   */
  public function getLimitB(): ?int {
    return $this->limit_b;
  }

  /**
   * {@inheritdoc}
   */
  public function isLimitActiveOnly(): bool {
    return (bool) $this->limit_active_only;
  }

  /**
   * {@inheritdoc}
   */
  public function getValidContactsA(): array {
    return $this->valid_contacts_a ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getValidContactsB(): array {
    return $this->valid_contacts_b ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function isReadonlyContactA(): bool {
    return (bool) ($this->readonly_contact_a ?? TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function isReadonlyContactB(): bool {
    return (bool) ($this->readonly_contact_b ?? TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function save() {
    if (!$this->get('asymmetric')) {
      $this->set('label_b', $this->get('label_a'));
      $this->set('label_b_plural', $this->get('label_a_plural'));
      $this->set('contact_type_b', $this->get('contact_type_a'));
      $this->set('limit_b', $this->get('limit_a'));
      $this->set('valid_contacts_b', $this->get('valid_contacts_a'));
      $this->set('readonly_contact_b', $this->get('readonly_contact_a'));
    }

    return parent::save();
  }

}
