<?php

declare(strict_types=1);

namespace Drupal\crm\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crm\CrmContactInterface;
use Drupal\crm\CrmUserContactMappingInterface;
use Drupal\crm\Form\UserContactMappingForm;
use Drupal\crm\UserContactMappingListBuilder;
use Drupal\crm\UserContactMappingStorageSchema;
use Drupal\crm\UserContactMappingViewBuilder;
use Drupal\user\UserInterface;
use Drupal\views\EntityViewsData;
use Drupal\Core\Entity\ContentEntityDeleteForm;

/**
 * Defines the user contact mapping entity class.
 */
#[ContentEntityType(
  id: 'crm_user_contact_mapping',
  label: new TranslatableMarkup('CRM User Contact Mapping'),
  label_collection: new TranslatableMarkup('CRM User Contact Mappings'),
  handlers: [
    'storage_schema' => UserContactMappingStorageSchema::class,
    'view_builder' => UserContactMappingViewBuilder::class,
    'list_builder' => UserContactMappingListBuilder::class,
    'views_data' => EntityViewsData::class,
    'form' => [
      'default' => UserContactMappingForm::class,
      'delete' => ContentEntityDeleteForm::class,
    ],
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
  base_table: 'crm_user_contact_mapping',
  translatable: FALSE,
  admin_permission: 'administer crm',
  entity_keys: [
    'id' => 'id',
    'uuid' => 'uuid',
  ],
  links: [
    'add-form' => '/admin/config/crm/user/add',
    'canonical' => '/admin/config/crm/user/{crm_user_contact_mapping}',
    'edit-form' => '/admin/config/crm/user/{crm_user_contact_mapping}/edit',
    'delete-form' => '/admin/config/crm/user/{crm_user_contact_mapping}/delete',
    'collection' => '/admin/config/crm/user/list',
  ],
)]
class UserContactMapping extends ContentEntityBase implements CrmUserContactMappingInterface {

  /**
   * {@inheritdoc}
   */
  public function getUser() {
    return $this->get('user')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getUserId() {
    return $this->get('user')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setUserId($uid) {
    $this->set('user', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setUser(UserInterface $account) {
    $this->set('user', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getContact() {
    return $this->get('crm_contact')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getContactId() {
    return $this->get('crm_contact')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setContactId($contact_id) {
    $this->set('crm_contact', $contact_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setContact(CrmContactInterface $person) {
    $this->set('crm_contact', $person->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user'] = BaseFieldDefinition::create('entity_reference')
      ->setRequired(TRUE)
      ->setCardinality(1)
      // @todo Replace with what ever would work with entity reference
      // from core. https://www.drupal.org/project/drupal/issues/2973455
      ->addConstraint('UniqueReference')
      ->setLabel(t('User'))
      ->setDescription(t('The user ID of the relation.'))
      ->setSetting('target_type', 'user')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'entity_reference_label',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['crm_contact'] = BaseFieldDefinition::create('entity_reference')
      ->setRequired(TRUE)
      ->setCardinality(1)
      // @todo Replace with what ever would work with entity reference
      // from core. https://www.drupal.org/project/drupal/issues/2973455
      ->addConstraint('UniqueReference')
      ->setLabel(t('Person'))
      ->setDescription(t('The person ID of the relation.'))
      ->setSetting('target_type', 'crm_contact')
      ->setSetting('handler_settings', ['target_bundles' => ['person' => 'person']])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'entity_reference_label',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created on'))
      ->setDescription(t('The time that the contact was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the contact was last edited.'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    $user = $this->getUser();
    $contact = $this->getContact();

    return ($user ? $user->getAccountName() : 'Unknown User') . ' - ' . ($contact ? $contact->label() : 'Unknown Contact');
  }

}
