<?php

namespace Drupal\crm\Form;

use Drupal\Core\Entity\BundleEntityFormBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for contact type forms.
 */
class ContactTypeForm extends BundleEntityFormBase {

  const SAVED_NEW = 1;
  const SAVED_UPDATED = 2;

  /**
   * The entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $storage;

  /**
   * Constructs a ContactTypeForm object.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage.
   */
  public function __construct(EntityStorageInterface $storage) {
    $this->storage = $storage;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')->getStorage('crm_relationship_type'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $entity_type = $this->entity;

    $form['label'] = [
      '#title' => $this->t('Label'),
      '#type' => 'textfield',
      '#default_value' => $entity_type->label(),
      '#description' => $this->t('The human-readable name of this contact type.'),
      '#required' => TRUE,
      '#size' => 30,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity_type->id(),
      '#maxlength' => EntityTypeInterface::BUNDLE_MAX_LENGTH,
      '#machine_name' => [
        'exists' => ['Drupal\crm\Entity\ContactType', 'load'],
        'source' => ['label'],
      ],
      '#description' => $this->t('A unique machine-readable name for this contact type. It must only contain lowercase letters, numbers, and underscores.'),
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#description' => $this->t('A short description of the contact.'),
      '#default_value' => $entity_type->get('description') ?: '',
    ];
    $form['additional_settings'] = [
      '#type' => 'vertical_tabs',
    ];

    $date = $entity_type->get('date');
    $form['start_date'] = [
      '#type' => 'details',
      '#title' => $this->t('Start Date'),
      '#group' => 'additional_settings',
    ];

    $form['start_date']['start_date_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Start Date Label'),
      '#description' => $this->t('The label for the start date field.'),
      '#default_value' => isset($date['start_date']) ? $date['start_date']['label'] : '',
    ];

    $form['start_date']['start_date_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Start Date Description'),
      '#description' => $this->t('The description for the start date field.'),
      '#default_value' => isset($date['start_date']) ? $date['start_date']['description'] : '',
    ];

    $form['end_date'] = [
      '#type' => 'details',
      '#title' => $this->t('End Date'),
      '#group' => 'additional_settings',
    ];

    $form['end_date']['end_date_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('End Date Label'),
      '#description' => $this->t('The label for the end date field.'),
      '#default_value' => isset($date['end_date']) ? $date['end_date']['label'] : '',
    ];

    $form['end_date']['end_date_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('End Date Description'),
      '#description' => $this->t('The description for the end date field.'),
      '#default_value' => isset($date['end_date']) ? $date['end_date']['description'] : '',
    ];

    if ($this->operation == 'edit') {
      $form['#title'] = $this->t('Edit %label contact type', ['%label' => $entity_type->label()]);
    }

    return $this->protectBundleIdElement($form);
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);
    $actions['submit']['#value'] = $this->t('Save contact type');
    $actions['delete']['#value'] = $this->t('Delete contact type');

    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\crm\Entity\ContactType $entity_type */
    $entity_type = $this->entity;

    $entity_type->set('id', trim($entity_type->id()));
    $entity_type->set('label', trim($form_state->getValue('label')));
    $entity_type->set('description', trim($form_state->getValue('description')));
    $date = [
      'start_date' => [
        'label' => $form_state->getValue('start_date_label'),
        'description' => $form_state->getValue('start_date_description'),
      ],
      'end_date' => [
        'label' => $form_state->getValue('end_date_label'),
        'description' => $form_state->getValue('end_date_description'),
      ],
    ];
    $entity_type->set('date', $date);

    $status = parent::save($form, $form_state);
    $t_args = ['%name' => $entity_type->label()];
    if ($status == self::SAVED_UPDATED) {
      $message = $this->t('The contact type %name has been updated.', $t_args);
      $this->messenger()->addStatus($message);
    }
    elseif ($status == self::SAVED_NEW) {
      $message = $this->t('The contact type %name has been added.', $t_args);
      $this->messenger()->addStatus($message);
    }

    $form_state->setRedirectUrl($entity_type->toUrl('collection'));

    return $status;
  }

  /**
   * Get the relationship type options.
   *
   * @return array
   *   An array of relationship type options.
   */
  private function getRelationshipTypeOptions() {
    // Load all relationship types and filter in PHP since config queries
    // don't support querying array values directly.
    $relationship_types = $this->storage->loadMultiple();
    $contact_type_id = $this->entity->id();
    $options = [];

    foreach ($relationship_types as $relationship_type) {
      // Normalize contact_type_a to array.
      $contact_type_a = $relationship_type->get('contact_type_a');
      if (!is_array($contact_type_a)) {
        $contact_type_a = $contact_type_a !== NULL && $contact_type_a !== '' ? [$contact_type_a] : [];
      }

      // Normalize contact_type_b to array.
      $contact_type_b = $relationship_type->get('contact_type_b');
      if (!is_array($contact_type_b)) {
        $contact_type_b = $contact_type_b !== NULL && $contact_type_b !== '' ? [$contact_type_b] : [];
      }

      // Check if contact type is in contact_type_a array.
      $is_a = in_array($contact_type_id, $contact_type_a, TRUE);
      if ($is_a) {
        $options[$relationship_type->id()] = $relationship_type->label();
      }
      elseif (in_array($contact_type_id, $contact_type_b, TRUE)) {
        $options[$relationship_type->id()] = $relationship_type->get('label_b');
      }
    }

    return $options;
  }

}
