<?php

namespace Drupal\crm\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Form controller for the crm relationship entity edit forms.
 */
class RelationshipForm extends ContentEntityForm {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a RelationshipForm object.
   *
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface|null $entity_type_bundle_info
   *   The entity type bundle service.
   * @param \Symfony\Component\HttpFoundation\RequestStack|null $request_stack
   *   The request stack.
   */
  public function __construct(
    EntityRepositoryInterface $entity_repository,
    TimeInterface $time,
    DateFormatterInterface $date_formatter,
    AccountInterface $current_user,
    EntityTypeManagerInterface $entity_type_manager,
    ?EntityTypeBundleInfoInterface $entity_type_bundle_info = NULL,
    ?RequestStack $request_stack = NULL,
  ) {

    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
    $this->dateFormatter = $date_formatter;
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entity_type_manager;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new self(
      $container->get('entity.repository'),
      $container->get('datetime.time'),
      $container->get('date.formatter'),
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('request_stack'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);
    $relationship = $this->getEntity();
    $bundle = $relationship->bundle->entity;

    // Normalize contact_type_a to array and convert to target_bundles format.
    $contact_type_a = $bundle->get('contact_type_a');
    if (!is_array($contact_type_a)) {
      $contact_type_a = $contact_type_a !== NULL && $contact_type_a !== '' ? [$contact_type_a] : [];
    }
    $target_bundles_a = array_combine($contact_type_a, $contact_type_a);
    $form['contact_a']['widget'][0]['target_id']['#selection_settings']['target_bundles'] = $target_bundles_a;
    $form['contact_a']['widget'][0]['target_id']['#title'] = $bundle->get('label_a');

    // Apply valid contacts restriction for Contact A.
    $this->applyValidContactConstraints($form, $relationship, $bundle, 'contact_a');

    // Normalize contact_type_b to array and convert to target_bundles format.
    $contact_type_b = $bundle->get('contact_type_b');
    if (!is_array($contact_type_b)) {
      $contact_type_b = $contact_type_b !== NULL && $contact_type_b !== '' ? [$contact_type_b] : [];
    }
    $target_bundles_b = array_combine($contact_type_b, $contact_type_b);
    $form['contact_b']['widget'][0]['target_id']['#selection_settings']['target_bundles'] = $target_bundles_b;
    $form['contact_b']['widget'][0]['target_id']['#title'] = $bundle->get('label_b');

    // Apply valid contacts restriction for Contact B.
    $this->applyValidContactConstraints($form, $relationship, $bundle, 'contact_b');

    // Hide contact fields if read-only and the contact is already set.
    if (!$relationship->isNew()) {
      if ($bundle->isReadonlyContactA() && $relationship->get('contact_a')->entity) {
        $form['contact_a']['#access'] = FALSE;
      }
      if ($bundle->isReadonlyContactB() && $relationship->get('contact_b')->entity) {
        $form['contact_b']['#access'] = FALSE;
      }
    }

    // Pre-populate contact fields from query parameters if creating new.
    if ($relationship->isNew() && $this->requestStack) {
      $request = $this->requestStack->getCurrentRequest();
      $contact_storage = $this->entityTypeManager->getStorage('crm_contact');

      // Check for contact_a query parameter.
      $contact_a_id = $request->query->get('contact_a');
      if ($contact_a_id) {
        $contact = $contact_storage->load($contact_a_id);
        if ($contact) {
          $form['contact_a']['widget'][0]['target_id']['#default_value'] = $contact;
          $form['contact_a']['widget']['#disabled'] = TRUE;
        }
      }

      // Check for contact_b query parameter.
      $contact_b_id = $request->query->get('contact_b');
      if ($contact_b_id) {
        $contact = $contact_storage->load($contact_b_id);
        if ($contact) {
          $form['contact_b']['widget'][0]['target_id']['#default_value'] = $contact;
          $form['contact_b']['widget']['#disabled'] = TRUE;
        }
      }
    }

    $form['advanced']['#attributes']['class'][] = 'entity-meta';

    $form['meta'] = [
      '#type' => 'details',
      '#group' => 'advanced',
      '#weight' => -10,
      '#title' => $this->t('Status'),
      '#attributes' => ['class' => ['entity-meta__header']],
      '#tree' => TRUE,
      '#access' => $this->currentUser->hasPermission('administer crm'),
    ];
    $form['meta']['published'] = [
      '#type' => 'item',
      '#markup' => $relationship->get('status')->value ? $this->t('Active') : $this->t('Inactive'),
      '#access' => !$relationship->isNew(),
      '#wrapper_attributes' => ['class' => ['entity-meta__title']],
    ];
    $form['meta']['changed'] = [
      '#type' => 'item',
      '#title' => $this->t('Last saved'),
      '#markup' => !$relationship->isNew() ? $this->dateFormatter->format($relationship->getChangedTime(), 'short') : $this->t('Not saved yet'),
      '#wrapper_attributes' => ['class' => ['entity-meta__last-saved']],
    ];

    $form['meta']['status'] = &$form['status'];
    $form['meta']['status']['#weight'] = 100;
    unset($form['status']);

    if (isset($form['uid'])) {
      unset($form['uid']);
    }

    if (isset($form['created'])) {
      $form['created']['#weight'] = 200;
      $form['meta']['created'] = &$form['created'];
      unset($form['created']);
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $result = parent::save($form, $form_state);

    $relationship = $this->getEntity();

    $label = $relationship->label() ?? 'No label';
    $message_arguments = ['%label' => $label];
    $logger_arguments = [
      '%label' => $label,
      'link' => $relationship->toLink($this->t('View'))->toString(),
    ];

    switch ($result) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('New crm relationship %label has been created.', $message_arguments));
        $this->logger('crm')->notice('Created new crm relationship %label', $logger_arguments);
        break;

      case SAVED_UPDATED:
        $this->messenger()->addStatus($this->t('The crm relationship %label has been updated.', $message_arguments));
        $this->logger('crm')->notice('Updated crm relationship %label.', $logger_arguments);
        break;
    }

    $form_state->setRedirect('entity.crm_relationship.canonical', ['crm_relationship' => $relationship->id()]);

    return $result;
  }

  /**
   * Applies valid contact constraints to a contact field.
   *
   * @param array &$form
   *   The form array.
   * @param \Drupal\crm\CrmRelationshipInterface $relationship
   *   The relationship entity.
   * @param \Drupal\crm\CrmRelationshipTypeInterface $bundle
   *   The relationship type entity.
   * @param string $field_name
   *   The field name ('contact_a' or 'contact_b').
   */
  protected function applyValidContactConstraints(array &$form, $relationship, $bundle, string $field_name): void {
    // Get valid contacts based on field name.
    $valid_contact_ids = $field_name === 'contact_a'
      ? $bundle->getValidContactsA()
      : $bundle->getValidContactsB();

    // If no valid contacts are configured, no restrictions apply.
    if (empty($valid_contact_ids)) {
      return;
    }

    // Load the valid contacts to verify they exist.
    $valid_contacts = $this->entityTypeManager
      ->getStorage('crm_contact')
      ->loadMultiple($valid_contact_ids);

    // Filter to only existing contacts.
    $existing_valid_ids = array_keys($valid_contacts);

    if (empty($existing_valid_ids)) {
      return;
    }

    // If exactly one valid contact exists, auto-select and lock the field.
    if (count($existing_valid_ids) === 1) {
      $single_contact = reset($valid_contacts);

      // Set the default value if not already set.
      if ($relationship->isNew() || empty($form[$field_name]['widget'][0]['target_id']['#default_value'])) {
        $form[$field_name]['widget'][0]['target_id']['#default_value'] = $single_contact;
      }

      // Disable the field to prevent changes.
      $form[$field_name]['widget']['#disabled'] = TRUE;

      // Add description explaining why it's locked.
      $form[$field_name]['widget'][0]['target_id']['#description'] = $this->t('This contact is automatically selected because it is the only valid option for this relationship type.');
    }
    else {
      // Multiple valid contacts, use custom selection handler to restrict.
      $form[$field_name]['widget'][0]['target_id']['#selection_handler'] = 'valid_contacts:crm_contact';
      $form[$field_name]['widget'][0]['target_id']['#selection_settings']['valid_contact_ids'] = $existing_valid_ids;
    }
  }

}
