<?php

namespace Drupal\crm\Form;

use Drupal\Core\Entity\BundleEntityFormBase;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for crm relationship type forms.
 */
class RelationshipTypeForm extends BundleEntityFormBase {

  const SAVED_NEW = 1;

  const SAVED_UPDATED = 2;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $entityTypeBundleInfo;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a RelationshipTypeForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle info service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeBundleInfoInterface $entity_type_bundle_info, EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new self(
      $container->get('entity_type.bundle.info'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $entity_type = $this->entity;
    if ($this->operation == 'edit') {
      $form['#title'] = $this->t('Edit %label crm relationship type', ['%label' => $entity_type->label()]);
    }

    $form['label'] = [
      '#title' => $this->t('Label'),
      '#type' => 'textfield',
      '#default_value' => $entity_type->label(),
      '#description' => $this->t('The human-readable name of this crm relationship type.'),
      '#required' => TRUE,
      '#size' => 30,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity_type->id(),
      '#maxlength' => EntityTypeInterface::BUNDLE_MAX_LENGTH,
      '#machine_name' => [
        'exists' => ['Drupal\crm\Entity\RelationshipType', 'load'],
        'source' => ['label'],
      ],
      '#description' => $this->t('A unique machine-readable name for this crm relationship type. It must only contain lowercase letters, numbers, and underscores.'),
    ];

    $form['description'] = [
      '#title' => $this->t('Description'),
      '#type' => 'textarea',
      '#default_value' => $entity_type->get('description'),
      '#description' => $this->t('A description of this crm relationship type.'),
    ];

    $form['asymmetric'] = [
      '#title' => $this->t('Asymmetric'),
      '#type' => 'checkbox',
      '#default_value' => $entity_type->isNew() ? 1 : $entity_type->get('asymmetric'),
      '#description' => $this->t('Check this box if the relationship is asymmetric.'),
    ];

    $form['label_a'] = [
      '#title' => $this->t('Contact A label'),
      '#type' => 'textfield',
      '#default_value' => $entity_type->get('label_a'),
      '#description' => $this->t('The human-readable name of this crm relationship type from contact A to contact B.'),
      '#required' => TRUE,
      '#size' => 30,
    ];

    $form['label_a_plural'] = [
      '#title' => $this->t('Contact A label (plural)'),
      '#type' => 'textfield',
      '#default_value' => $entity_type->get('label_a_plural'),
      '#description' => $this->t('The plural form of the Contact A label.'),
      '#size' => 30,
    ];

    // Normalize contact_type_a to array for default value.
    $contact_type_a_default = $entity_type->get('contact_type_a');
    if (!is_array($contact_type_a_default)) {
      $contact_type_a_default = $contact_type_a_default !== NULL && $contact_type_a_default !== '' ? [$contact_type_a_default] : [];
    }

    $form['contact_type_a'] = [
      '#title' => $this->t('Contact A type'),
      '#type' => 'select',
      '#options' => $this->getContactTypeOptions(),
      '#default_value' => $contact_type_a_default,
      '#description' => $this->t('The contact type for the first contact in the relationship.'),
      '#required' => TRUE,
      '#multiple' => TRUE,
    ];

    $form['label_b'] = [
      '#title' => $this->t('Contact B label'),
      '#type' => 'textfield',
      '#default_value' => $entity_type->get('label_b'),
      '#description' => $this->t('The human-readable name of this crm relationship type from contact A to contact B.'),
      '#required' => TRUE,
      '#size' => 30,
      '#states' => [
        'visible' => [
          ':input[name="asymmetric"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['label_b_plural'] = [
      '#title' => $this->t('Contact B label (plural)'),
      '#type' => 'textfield',
      '#default_value' => $entity_type->get('label_b_plural'),
      '#description' => $this->t('The plural form of the Contact B label.'),
      '#size' => 30,
      '#states' => [
        'visible' => [
          ':input[name="asymmetric"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Normalize contact_type_b to array for default value.
    $contact_type_b_default = $entity_type->get('contact_type_b');
    if (!is_array($contact_type_b_default)) {
      $contact_type_b_default = $contact_type_b_default !== NULL && $contact_type_b_default !== '' ? [$contact_type_b_default] : [];
    }

    $form['contact_type_b'] = [
      '#title' => $this->t('Contact B type'),
      '#type' => 'select',
      '#options' => $this->getContactTypeOptions(),
      '#default_value' => $contact_type_b_default,
      '#description' => $this->t('The contact type for the second contact in the relationship.'),
      '#required' => TRUE,
      '#multiple' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="asymmetric"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Vertical tabs for additional settings.
    $form['additional_settings'] = [
      '#type' => 'vertical_tabs',
      '#weight' => 99,
    ];

    // Attach JavaScript for vertical tab summaries.
    $form['#attached']['library'][] = 'crm/crm.relationship_type_form';

    // Relationship limits section.
    $form['limits'] = [
      '#type' => 'details',
      '#title' => $this->t('Relationship limits'),
      '#group' => 'additional_settings',
    ];

    $form['limits']['limit_a'] = [
      '#title' => $this->t('Contact A limit'),
      '#type' => 'number',
      '#default_value' => $entity_type->get('limit_a'),
      '#description' => $this->t('Maximum number of relationships of this type where a contact can be in the Contact A position. Leave empty for unlimited.'),
      '#min' => 1,
    ];

    $form['limits']['limit_b'] = [
      '#title' => $this->t('Contact B limit'),
      '#type' => 'number',
      '#default_value' => $entity_type->get('limit_b'),
      '#description' => $this->t('Maximum number of relationships of this type where a contact can be in the Contact B position. Leave empty for unlimited.'),
      '#min' => 1,
      '#states' => [
        'visible' => [
          ':input[name="asymmetric"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['limits']['limit_active_only'] = [
      '#title' => $this->t('Count active relationships only'),
      '#type' => 'checkbox',
      '#default_value' => $entity_type->get('limit_active_only') ?? FALSE,
      '#description' => $this->t('If checked, only active relationships will count toward the limit. Inactive relationships will be ignored.'),
    ];

    // Valid contacts section.
    $form['valid_contacts'] = [
      '#type' => 'details',
      '#title' => $this->t('Valid contacts'),
      '#group' => 'additional_settings',
      '#description' => $this->t('Optionally restrict which contacts can be selected for each side of the relationship. Leave empty for no restrictions.'),
    ];

    // Get valid contacts A default value.
    $valid_contacts_a_default = [];
    $valid_contacts_a_ids = $entity_type->getValidContactsA();
    if (!empty($valid_contacts_a_ids)) {
      $valid_contacts_a_default = $this->entityTypeManager
        ->getStorage('crm_contact')
        ->loadMultiple($valid_contacts_a_ids);
    }

    $form['valid_contacts']['valid_contacts_a'] = [
      '#title' => $this->t('Valid contacts for Contact A'),
      '#type' => 'entity_autocomplete',
      '#target_type' => 'crm_contact',
      '#default_value' => $valid_contacts_a_default,
      '#tags' => TRUE,
      '#description' => $this->t('Select the contacts that are valid for Contact A. If only one contact is selected, it will be automatically selected and locked when creating relationships.'),
    ];

    // Get valid contacts B default value.
    $valid_contacts_b_default = [];
    $valid_contacts_b_ids = $entity_type->getValidContactsB();
    if (!empty($valid_contacts_b_ids)) {
      $valid_contacts_b_default = $this->entityTypeManager
        ->getStorage('crm_contact')
        ->loadMultiple($valid_contacts_b_ids);
    }

    $form['valid_contacts']['valid_contacts_b'] = [
      '#title' => $this->t('Valid contacts for Contact B'),
      '#type' => 'entity_autocomplete',
      '#target_type' => 'crm_contact',
      '#default_value' => $valid_contacts_b_default,
      '#tags' => TRUE,
      '#description' => $this->t('Select the contacts that are valid for Contact B. If only one contact is selected, it will be automatically selected and locked when creating relationships.'),
      '#states' => [
        'visible' => [
          ':input[name="asymmetric"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Contact readonly section.
    $form['contact_readonly'] = [
      '#type' => 'details',
      '#title' => $this->t('Readonly'),
      '#group' => 'additional_settings',
      '#description' => $this->t('Control whether contacts can be changed after being set.'),
    ];

    $form['contact_readonly']['readonly_contact_a'] = [
      '#title' => $this->t('Make Contact A read-only after being set'),
      '#type' => 'checkbox',
      '#default_value' => $entity_type->isNew() ? TRUE : $entity_type->isReadonlyContactA(),
      '#description' => $this->t('If checked, Contact A cannot be changed after the relationship is created.'),
    ];

    $form['contact_readonly']['readonly_contact_b'] = [
      '#title' => $this->t('Make Contact B read-only after being set'),
      '#type' => 'checkbox',
      '#default_value' => $entity_type->isNew() ? TRUE : $entity_type->isReadonlyContactB(),
      '#description' => $this->t('If checked, Contact B cannot be changed after the relationship is created.'),
      '#states' => [
        'visible' => [
          ':input[name="asymmetric"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $this->protectBundleIdElement($form);
  }

  /**
   * Returns a list of contact types.
   */
  protected function getContactTypeOptions() {
    $crm_contact_type = $this->entityTypeBundleInfo->getBundleInfo('crm_contact');
    $options = [];
    foreach ($crm_contact_type as $type => $contact) {
      $options[$type] = $contact['label'];
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Only validate contact type removal on edit, not add.
    if ($this->entity->isNew()) {
      return;
    }

    // Get old and new contact types.
    $old_types_a = $this->entity->get('contact_type_a') ?? [];
    $new_types_a = $form_state->getValue('contact_type_a') ?? [];
    if (!is_array($new_types_a)) {
      $new_types_a = $new_types_a !== NULL && $new_types_a !== '' ? [$new_types_a] : [];
    }
    $new_types_a = array_values(array_filter($new_types_a));

    $old_types_b = $this->entity->get('contact_type_b') ?? [];
    $new_types_b = $form_state->getValue('contact_type_b') ?? [];
    if (!is_array($new_types_b)) {
      $new_types_b = $new_types_b !== NULL && $new_types_b !== '' ? [$new_types_b] : [];
    }
    $new_types_b = array_values(array_filter($new_types_b));

    // Find removed contact types.
    $removed_types_a = array_diff($old_types_a, $new_types_a);
    $removed_types_b = array_diff($old_types_b, $new_types_b);

    // Check if any removed types are still in use.
    foreach ($removed_types_a as $removed_type) {
      if ($this->contactTypeInUse($removed_type, 0)) {
        $form_state->setErrorByName('contact_type_a', $this->t('Cannot remove contact type %type from Contact A because existing relationships use it.', [
          '%type' => $removed_type,
        ]));
      }
    }

    foreach ($removed_types_b as $removed_type) {
      if ($this->contactTypeInUse($removed_type, 1)) {
        $form_state->setErrorByName('contact_type_b', $this->t('Cannot remove contact type %type from Contact B because existing relationships use it.', [
          '%type' => $removed_type,
        ]));
      }
    }
  }

  /**
   * Checks if a contact type is in use by existing relationships.
   *
   * @param string $contact_type
   *   The contact type (bundle) to check.
   * @param int $delta
   *   The delta position (0 for contact_a, 1 for contact_b).
   *
   * @return bool
   *   TRUE if the contact type is in use, FALSE otherwise.
   */
  protected function contactTypeInUse(string $contact_type, int $delta): bool {
    $relationship_type_id = $this->entity->id();

    // Query relationships of this type.
    $relationship_ids = $this->entityTypeManager
      ->getStorage('crm_relationship')
      ->getQuery()
      ->condition('bundle', $relationship_type_id)
      ->accessCheck(FALSE)
      ->execute();

    if (empty($relationship_ids)) {
      return FALSE;
    }

    // Load relationships and check contact types at the specified delta.
    $relationships = $this->entityTypeManager
      ->getStorage('crm_relationship')
      ->loadMultiple($relationship_ids);

    foreach ($relationships as $relationship) {
      $contacts = $relationship->get('contacts')->referencedEntities();
      if (isset($contacts[$delta]) && $contacts[$delta]->bundle() === $contact_type) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);
    $actions['submit']['#value'] = $this->t('Save relationship type');

    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $entity_type = $this->entity;
    $entity_type
      ->set('id', trim($entity_type->id()))
      ->set('label', trim($entity_type->label()));

    // Normalize contact_type_a to array.
    $contact_type_a = $form_state->getValue('contact_type_a');
    if (!is_array($contact_type_a)) {
      $contact_type_a = $contact_type_a !== NULL && $contact_type_a !== '' ? [$contact_type_a] : [];
    }
    $entity_type->set('contact_type_a', array_values(array_filter($contact_type_a)));

    // Normalize contact_type_b to array.
    $contact_type_b = $form_state->getValue('contact_type_b');
    if (!is_array($contact_type_b)) {
      $contact_type_b = $contact_type_b !== NULL && $contact_type_b !== '' ? [$contact_type_b] : [];
    }
    $entity_type->set('contact_type_b', array_values(array_filter($contact_type_b)));

    // Set limit values, converting empty strings to NULL.
    $limit_a = $form_state->getValue('limit_a');
    $entity_type->set('limit_a', $limit_a !== '' && $limit_a !== NULL ? (int) $limit_a : NULL);

    $limit_b = $form_state->getValue('limit_b');
    $entity_type->set('limit_b', $limit_b !== '' && $limit_b !== NULL ? (int) $limit_b : NULL);

    $entity_type->set('limit_active_only', (bool) $form_state->getValue('limit_active_only'));

    // Normalize valid_contacts_a from entity_autocomplete format.
    $valid_contacts_a = $form_state->getValue('valid_contacts_a') ?? [];
    $valid_contacts_a_ids = [];
    if (!empty($valid_contacts_a)) {
      foreach ($valid_contacts_a as $item) {
        if (is_array($item) && isset($item['target_id'])) {
          $valid_contacts_a_ids[] = (int) $item['target_id'];
        }
        elseif (is_numeric($item)) {
          $valid_contacts_a_ids[] = (int) $item;
        }
      }
    }
    $entity_type->set('valid_contacts_a', $valid_contacts_a_ids);

    // Normalize valid_contacts_b from entity_autocomplete format.
    $valid_contacts_b = $form_state->getValue('valid_contacts_b') ?? [];
    $valid_contacts_b_ids = [];
    if (!empty($valid_contacts_b)) {
      foreach ($valid_contacts_b as $item) {
        if (is_array($item) && isset($item['target_id'])) {
          $valid_contacts_b_ids[] = (int) $item['target_id'];
        }
        elseif (is_numeric($item)) {
          $valid_contacts_b_ids[] = (int) $item;
        }
      }
    }
    $entity_type->set('valid_contacts_b', $valid_contacts_b_ids);

    // Set readonly contact settings.
    $entity_type->set('readonly_contact_a', (bool) $form_state->getValue('readonly_contact_a'));
    $entity_type->set('readonly_contact_b', (bool) $form_state->getValue('readonly_contact_b'));

    if (!$entity_type->get('asymmetric')) {
      $entity_type->set('label_b', $entity_type->get('label_a'));
      $entity_type->set('label_b_plural', $entity_type->get('label_a_plural'));
      $entity_type->set('contact_type_b', $entity_type->get('contact_type_a'));
      $entity_type->set('limit_b', $entity_type->get('limit_a'));
      $entity_type->set('valid_contacts_b', $entity_type->get('valid_contacts_a'));
      $entity_type->set('readonly_contact_b', $entity_type->get('readonly_contact_a'));
    }

    $status = $entity_type->save();

    $t_args = ['%name' => $entity_type->label()];
    if ($status == self::SAVED_UPDATED) {
      $message = $this->t('The crm relationship type %name has been updated.', $t_args);
    }
    elseif ($status == self::SAVED_NEW) {
      $message = $this->t('The crm relationship type %name has been added.', $t_args);
    }
    $this->messenger()->addStatus($message);

    $form_state->setRedirectUrl($entity_type->toUrl('collection'));

    return $status;
  }

}
