<?php

declare(strict_types=1);

namespace Drupal\crm\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\crm\CrmRelationshipInterface;
use Drupal\crm\RelationshipStatisticsServiceInterface;

/**
 * Hooks relating to relationship entities.
 */
class RelationshipHooks {

  /**
   * Constructs a RelationshipHooks object.
   *
   * @param \Drupal\crm\RelationshipStatisticsServiceInterface $statisticsService
   *   The relationship statistics service.
   */
  public function __construct(
    protected RelationshipStatisticsServiceInterface $statisticsService,
  ) {}

  /**
   * Implements hook_ENTITY_TYPE_insert() for crm_relationship.
   */
  #[Hook('crm_relationship_insert')]
  public function relationshipInsert(CrmRelationshipInterface $relationship): void {
    // Only update statistics for active relationships.
    if (!$relationship->get('status')->value) {
      return;
    }

    $contacts = $this->getRelationshipContacts($relationship);

    if ($contacts['a']) {
      $type_key = $this->statisticsService->getTypeKey($relationship, 'a');
      $this->statisticsService->increment($contacts['a'], $type_key);
    }

    if ($contacts['b']) {
      $type_key = $this->statisticsService->getTypeKey($relationship, 'b');
      $this->statisticsService->increment($contacts['b'], $type_key);
    }
  }

  /**
   * Implements hook_ENTITY_TYPE_update() for crm_relationship.
   */
  #[Hook('crm_relationship_update')]
  public function relationshipUpdate(CrmRelationshipInterface $relationship): void {
    $original = $relationship->original ?? NULL;

    $current_status = (bool) $relationship->get('status')->value;
    $original_status = $original ? (bool) $original->get('status')->value : FALSE;

    $current_contacts = $this->getRelationshipContacts($relationship);
    $original_contacts = $original ? $this->getRelationshipContacts($original) : ['a' => NULL, 'b' => NULL];

    // Handle status changes.
    if ($current_status !== $original_status) {
      if ($current_status) {
        // Relationship became active - increment.
        $this->incrementForRelationship($relationship, $current_contacts);
      }
      else {
        // Relationship became inactive - decrement.
        $this->decrementForRelationship($relationship, $current_contacts);
      }
      return;
    }

    // If relationship is inactive, no statistics changes needed.
    if (!$current_status) {
      return;
    }

    // Handle contact reference changes.
    $this->handleContactChanges($relationship, $original_contacts, $current_contacts);
  }

  /**
   * Implements hook_ENTITY_TYPE_delete() for crm_relationship.
   */
  #[Hook('crm_relationship_delete')]
  public function relationshipDelete(CrmRelationshipInterface $relationship): void {
    // Only update statistics for active relationships.
    if (!$relationship->get('status')->value) {
      return;
    }

    $contacts = $this->getRelationshipContacts($relationship);

    if ($contacts['a']) {
      $type_key = $this->statisticsService->getTypeKey($relationship, 'a');
      $this->statisticsService->decrement($contacts['a'], $type_key);
    }

    if ($contacts['b']) {
      $type_key = $this->statisticsService->getTypeKey($relationship, 'b');
      $this->statisticsService->decrement($contacts['b'], $type_key);
    }
  }

  /**
   * Gets the contact entities from a relationship.
   *
   * @param \Drupal\crm\CrmRelationshipInterface $relationship
   *   The relationship entity.
   *
   * @return array
   *   An array with 'a' and 'b' keys containing contact entities or NULL.
   */
  protected function getRelationshipContacts(CrmRelationshipInterface $relationship): array {
    $contacts_field = $relationship->get('contacts')->getValue();

    $contact_a = NULL;
    $contact_b = NULL;

    if (!empty($contacts_field[0]['target_id'])) {
      $contact_a = $relationship->get('contact_a')->entity;
    }

    if (!empty($contacts_field[1]['target_id'])) {
      $contact_b = $relationship->get('contact_b')->entity;
    }

    return [
      'a' => $contact_a,
      'b' => $contact_b,
    ];
  }

  /**
   * Increments statistics for both contacts in a relationship.
   *
   * @param \Drupal\crm\CrmRelationshipInterface $relationship
   *   The relationship entity.
   * @param array $contacts
   *   The contacts array with 'a' and 'b' keys.
   */
  protected function incrementForRelationship(CrmRelationshipInterface $relationship, array $contacts): void {
    if ($contacts['a']) {
      $type_key = $this->statisticsService->getTypeKey($relationship, 'a');
      $this->statisticsService->increment($contacts['a'], $type_key);
    }

    if ($contacts['b']) {
      $type_key = $this->statisticsService->getTypeKey($relationship, 'b');
      $this->statisticsService->increment($contacts['b'], $type_key);
    }
  }

  /**
   * Decrements statistics for both contacts in a relationship.
   *
   * @param \Drupal\crm\CrmRelationshipInterface $relationship
   *   The relationship entity.
   * @param array $contacts
   *   The contacts array with 'a' and 'b' keys.
   */
  protected function decrementForRelationship(CrmRelationshipInterface $relationship, array $contacts): void {
    if ($contacts['a']) {
      $type_key = $this->statisticsService->getTypeKey($relationship, 'a');
      $this->statisticsService->decrement($contacts['a'], $type_key);
    }

    if ($contacts['b']) {
      $type_key = $this->statisticsService->getTypeKey($relationship, 'b');
      $this->statisticsService->decrement($contacts['b'], $type_key);
    }
  }

  /**
   * Handles contact reference changes.
   *
   * @param \Drupal\crm\CrmRelationshipInterface $relationship
   *   The relationship entity.
   * @param array $original_contacts
   *   The original contacts array.
   * @param array $current_contacts
   *   The current contacts array.
   */
  protected function handleContactChanges(CrmRelationshipInterface $relationship, array $original_contacts, array $current_contacts): void {
    // Check if contact A changed.
    $original_a_id = $original_contacts['a'] ? $original_contacts['a']->id() : NULL;
    $current_a_id = $current_contacts['a'] ? $current_contacts['a']->id() : NULL;

    if ($original_a_id !== $current_a_id) {
      $type_key = $this->statisticsService->getTypeKey($relationship, 'a');

      // Decrement from old contact.
      if ($original_contacts['a']) {
        $this->statisticsService->decrement($original_contacts['a'], $type_key);
      }

      // Increment on new contact.
      if ($current_contacts['a']) {
        $this->statisticsService->increment($current_contacts['a'], $type_key);
      }
    }

    // Check if contact B changed.
    $original_b_id = $original_contacts['b'] ? $original_contacts['b']->id() : NULL;
    $current_b_id = $current_contacts['b'] ? $current_contacts['b']->id() : NULL;

    if ($original_b_id !== $current_b_id) {
      $type_key = $this->statisticsService->getTypeKey($relationship, 'b');

      // Decrement from old contact.
      if ($original_contacts['b']) {
        $this->statisticsService->decrement($original_contacts['b'], $type_key);
      }

      // Increment on new contact.
      if ($current_contacts['b']) {
        $this->statisticsService->increment($current_contacts['b'], $type_key);
      }
    }
  }

}
