<?php

namespace Drupal\crm\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountInterface;
use Drupal\crm\Event\CrmUserContactMappingEvent;
use Drupal\user\UserDataInterface;
use Drupal\user\UserInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\name\NameFormatterInterface;

/**
 * Hooks relating to users.
 */
class UserHooks {
  use StringTranslationTrait;

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EventDispatcherInterface $eventDispatcher,
    protected UserDataInterface $userData,
    protected AccountProxyInterface $currentUser,
    protected NameFormatterInterface $nameFormatter,
  ) {}

  /**
   * Implements hook_ENTITY_TYPE_insert().
   */
  #[Hook('user_insert')]
  public function userInsert(UserInterface $user) {

    $auto_create_crm_user_contact_mapping = $this->configFactory
      ->get('crm.user_contact_mapping.settings')
      ->get('auto_create_crm_user_contact_mapping');
    if (!$auto_create_crm_user_contact_mapping) {
      return;
    }

    $crm_user_contact_mapping = $this->entityTypeManager
      ->getStorage('crm_user_contact_mapping')
      ->create(['user' => $user->id()]);
    $event = new CrmUserContactMappingEvent($crm_user_contact_mapping);
    $this->eventDispatcher->dispatch($event, CrmUserContactMappingEvent::EVENT_NAME);
    $crm_user_contact_mapping = $event->getCrmUserContactMapping();
    $contact = $crm_user_contact_mapping->getContact();

    if ($contact == NULL) {
      return NULL;
    }
    if ($contact->get('bundle')->target_id != 'person') {
      return NULL;
    }

    $crm_user_contact_mapping->save();
  }

  /**
   * Implements hook_user_format_name_alter().
   */
  #[Hook('user_format_name_alter')]
  public function userContactMappingFormatNameAlter(&$name, AccountInterface $account) {
    $override = $this->configFactory->get('crm.user_contact_mapping.settings')->get('display_name');
    if (!$override) {
      return;
    }
    $crm_user_contact_mappings = $this->entityTypeManager
      ->getStorage('crm_user_contact_mapping')
      ->loadByProperties(['user' => $account->id()]);
    if ($crm_user_contact_mappings == NULL) {
      return;
    }
    /** @var \Drupal\crm\Entity\UserContactMapping */
    $crm_user_contact_mapping = reset($crm_user_contact_mappings);
    $contact = $crm_user_contact_mapping->getContact();
    $name = $contact->label();

    $user_name_format = $this->userData->get('crm', $account->id(), 'name_format');
    if (!$user_name_format) {
      return;
    }
    if (!$account->hasPermission('crm user alter display name')) {
      return;
    }

    $name_array = $contact->get('full_name')->getValue();
    if (empty($name_array)) {
      return;
    }
    $name_array = $name_array[0];
    $name_array['preferred'] = $contact->get('preferred_name')->value;
    $name_array['alternative'] = array_map(function ($alias) {
      return $alias['value'];
    }, $contact->get('aliases')->getValue());
    $name_array['alternative'] = implode(', ', $name_array['alternative']);

    $formatted_name = $this->nameFormatter->format($name_array, $user_name_format);

    $name = html_entity_decode($formatted_name, ENT_QUOTES | ENT_HTML5, 'UTF-8');

  }

  /**
   * Implements hook_form_FORM_ID_alter().
   */
  #[Hook('form_user_form_alter')]
  public function userFormAlter(&$form, FormStateInterface $form_state) {
    $user_contact_mapping_settings = $this->configFactory->get('crm.user_contact_mapping.settings');

    if (!$user_contact_mapping_settings->get('display_name')) {
      return;
    }

    $form_user = $form_state->getFormObject()->getEntity();

    $user_contact_mapping = $this->entityTypeManager
      ->getStorage('crm_user_contact_mapping')
      ->loadByProperties(['user' => $form_user->id()]);
    if ($user_contact_mapping == NULL) {
      return;
    }

    $user_contact_mapping = reset($user_contact_mapping);

    $same_user = $this->currentUser->id() == $form_user->id();
    if (!$same_user) {
      return;
    }

    if (!$this->currentUser->hasPermission('crm user alter display name')) {
      return;
    }

    $format_options = $this->entityTypeManager->getStorage('name_format')->loadMultiple();
    $format_options = array_map(function ($format) {
      return $format->label();
    }, $format_options);

    $person_contact_type = $user_contact_mapping->getContact()->get('bundle')->entity;
    $system_format = $person_contact_type->getThirdPartySetting('crm', 'name_format', 'default');
    $format_options = ['_none' => $this->t('System (@system)', ['@system' => $system_format])] + $format_options;

    $user_name_format = $this->userData->get('crm', $form_user->id(), 'name_format');

    $form['crm'] = [
      '#type' => 'details',
      '#title' => $this->t('CRM'),
      '#open' => TRUE,
    ];

    $form['crm']['name_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Name Format'),
      '#options' => $format_options,
      '#default_value' => $user_name_format,
    ];

    $form['actions']['submit']['#submit'][] = [static::class, 'userFormAlterSubmit'];
  }

  /**
   * Static submit handler for user form alter.
   */
  public static function userFormAlterSubmit(array &$form, FormStateInterface $form_state) {

    $form_user = $form_state->getFormObject()->getEntity();
    $user_name_format = $form_state->getValue('name_format');

    /** @var \Drupal\user\UserDataInterface $user_data */
    $user_data = \Drupal::service('user.data');

    if ($user_name_format == '_none') {
      $user_data->delete('crm', $form_user->id(), 'name_format');
      return;
    }
    $user_data->set('crm', $form_user->id(), 'name_format', $user_name_format);
  }

  /**
   * Implements hook_ENTITY_TYPE_delete().
   */
  #[Hook('user_delete')]
  public function userDelete(UserInterface $user) {
    $crm_user_contact_mappings = $this->entityTypeManager
      ->getStorage('crm_user_contact_mapping')
      ->loadByProperties(['user' => $user->id()]);
    if ($crm_user_contact_mappings != NULL) {
      $crm_user_contact_mapping = reset($crm_user_contact_mappings);
      $crm_user_contact_mapping->delete();
    }
  }

}
