<?php

declare(strict_types=1);

namespace Drupal\crm\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Formatter for the relationship statistics field.
 */
#[FieldFormatter(
  id: "crm_relationship_statistics_default",
  label: new TranslatableMarkup("Relationship Statistics"),
  description: new TranslatableMarkup("Display relationship type labels with counts."),
  field_types: ["crm_relationship_statistics"],
)]
class RelationshipStatisticsFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Cached relationship types.
   *
   * @var array|null
   */
  protected ?array $relationshipTypes = NULL;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'format' => 'label_count',
      'label_position' => 'opposite',
      'sort_by' => 'none',
      'sort_order' => 'asc',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = [];

    $elements['format'] = [
      '#type' => 'select',
      '#title' => $this->t('Display format'),
      '#options' => [
        'label_count' => $this->t('Label (count) - e.g., "Parent (3)"'),
        'label_colon_count' => $this->t('Label: count - e.g., "Parent: 3"'),
        'count_label' => $this->t('count Label - e.g., "3 Parents"'),
        'label_only' => $this->t('Label only'),
      ],
      '#default_value' => $this->getSetting('format'),
    ];

    $elements['label_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Label position'),
      '#description' => $this->t('For asymmetric relationships, choose which label to display.'),
      '#options' => [
        'opposite' => $this->t('Opposite position (show what the related contacts are)'),
        'same' => $this->t('Same position (show what this contact is in the relationship)'),
      ],
      '#default_value' => $this->getSetting('label_position'),
    ];

    $elements['sort_by'] = [
      '#type' => 'select',
      '#title' => $this->t('Sort by'),
      '#options' => [
        'none' => $this->t('No sorting (maintain original order)'),
        'label' => $this->t('Label'),
        'count' => $this->t('Count'),
      ],
      '#default_value' => $this->getSetting('sort_by'),
    ];

    $elements['sort_order'] = [
      '#type' => 'select',
      '#title' => $this->t('Sort order'),
      '#options' => [
        'asc' => $this->t('Ascending (A-Z or lowest first)'),
        'desc' => $this->t('Descending (Z-A or highest first)'),
      ],
      '#default_value' => $this->getSetting('sort_order'),
      '#states' => [
        'invisible' => [
          ':input[name="fields[relationship_statistics][settings_edit_form][settings][sort_by]"]' => ['value' => 'none'],
        ],
      ],
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $format = $this->getSetting('format');
    $formats = [
      'label_count' => $this->t('Label (count)'),
      'label_colon_count' => $this->t('Label: count'),
      'count_label' => $this->t('count Label'),
      'label_only' => $this->t('Label only'),
    ];
    $summary[] = $this->t('Format: @format', ['@format' => $formats[$format] ?? $format]);

    $label_position = $this->getSetting('label_position');
    $label_positions = [
      'opposite' => $this->t('Opposite position'),
      'same' => $this->t('Same position'),
    ];
    $summary[] = $this->t('Label: @position', ['@position' => $label_positions[$label_position] ?? $label_position]);

    $sort_by = $this->getSetting('sort_by');
    if ($sort_by !== 'none') {
      $sort_order = $this->getSetting('sort_order');
      $sort_options = [
        'label' => $this->t('label'),
        'count' => $this->t('count'),
      ];
      $order_options = [
        'asc' => $this->t('ascending'),
        'desc' => $this->t('descending'),
      ];
      $summary[] = $this->t('Sort by @sort_by (@order)', [
        '@sort_by' => $sort_options[$sort_by] ?? $sort_by,
        '@order' => $order_options[$sort_order] ?? $sort_order,
      ]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $format = $this->getSetting('format');
    $label_position = $this->getSetting('label_position');
    $sort_by = $this->getSetting('sort_by');
    $sort_order = $this->getSetting('sort_order');
    $use_opposite = $label_position === 'opposite';

    // Build an array of items with their labels and counts for sorting.
    $processed_items = [];
    foreach ($items as $delta => $item) {
      $type_key = $item->value;
      $count = (int) $item->count;
      // For 'label_only' format, always use singular label since count is not
      // displayed. For other formats, use singular/plural based on count.
      $label_count = $format === 'label_only' ? 1 : $count;
      $label = $this->getLabel($type_key, $use_opposite, $label_count);

      $processed_items[] = [
        'delta' => $delta,
        'label' => $label,
        'count' => $count,
      ];
    }

    // Apply sorting if configured.
    if ($sort_by !== 'none' && count($processed_items) > 1) {
      usort($processed_items, function ($a, $b) use ($sort_by, $sort_order) {
        if ($sort_by === 'label') {
          $comparison = strcasecmp($a['label'], $b['label']);
        }
        else {
          $comparison = $a['count'] <=> $b['count'];
        }

        return $sort_order === 'desc' ? -$comparison : $comparison;
      });
    }

    // Build the render elements.
    $elements = [];
    foreach ($processed_items as $index => $item) {
      $label = $item['label'];
      $count = $item['count'];

      $output = match ($format) {
        'label_count' => $this->t('@label (@count)', ['@label' => $label, '@count' => $count]),
        'label_colon_count' => $this->t('@label: @count', ['@label' => $label, '@count' => $count]),
        'count_label' => $this->t('@count @label', ['@count' => $count, '@label' => $label]),
        'label_only' => $label,
        default => $this->t('@label (@count)', ['@label' => $label, '@count' => $count]),
      };

      $elements[$index] = ['#markup' => $output];
    }

    return $elements;
  }

  /**
   * Gets the human-readable label for a relationship type key.
   *
   * @param string $type_key
   *   The relationship type key (e.g., "friends" or "parent_child:a").
   * @param bool $use_opposite
   *   If TRUE, return the opposite position's label (what the related contacts
   *   are). If FALSE, return the same position's label (what this contact is
   *   in the relationship). Defaults to TRUE.
   * @param int $count
   *   The count of relationships. Used to determine singular vs plural label.
   *   Defaults to 1 (singular).
   *
   * @return string
   *   The human-readable label.
   */
  protected function getLabel(string $type_key, bool $use_opposite = TRUE, int $count = 1): string {
    // Parse the type key to get the relationship type ID and position.
    $parts = explode(':', $type_key);
    $type_id = $parts[0];
    $position = $parts[1] ?? NULL;

    $relationship_types = $this->getRelationshipTypes();

    if (!isset($relationship_types[$type_id])) {
      // Fallback to the raw key if type not found.
      return $type_key;
    }

    $type = $relationship_types[$type_id];

    // Determine if we should use plural (count != 1).
    $use_plural = $count !== 1;

    // For asymmetric relationships, get the appropriate label based on
    // the use_opposite setting.
    if ($position === 'a') {
      // Position A: opposite means label_b, same means label_a.
      $label_key = $use_opposite ? 'label_b' : 'label_a';
      $plural_key = $use_opposite ? 'label_b_plural' : 'label_a_plural';
      $singular_label = $type->get($label_key) ?? $type->label();
      if ($use_plural) {
        $plural_label = $type->get($plural_key);
        return !empty($plural_label) ? $plural_label : $singular_label;
      }
      return $singular_label;
    }
    elseif ($position === 'b') {
      // Position B: opposite means label_a, same means label_b.
      $label_key = $use_opposite ? 'label_a' : 'label_b';
      $plural_key = $use_opposite ? 'label_a_plural' : 'label_b_plural';
      $singular_label = $type->get($label_key) ?? $type->label();
      if ($use_plural) {
        $plural_label = $type->get($plural_key);
        return !empty($plural_label) ? $plural_label : $singular_label;
      }
      return $singular_label;
    }

    // For symmetric relationships, use the main label.
    return $type->label();
  }

  /**
   * Gets all relationship types, cached for the request.
   *
   * @return array
   *   An array of relationship type entities keyed by ID.
   */
  protected function getRelationshipTypes(): array {
    if ($this->relationshipTypes === NULL) {
      $this->relationshipTypes = $this->entityTypeManager
        ->getStorage('crm_relationship_type')
        ->loadMultiple();
    }
    return $this->relationshipTypes;
  }

}
