<?php

declare(strict_types=1);

namespace Drupal\crm\Plugin\Field\FieldType;

use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Defines the 'crm_relationship_statistics' field type.
 *
 * This field stores relationship type identifiers and their counts.
 * It is automatically updated when relationships are created, updated,
 * or deleted.
 */
#[FieldType(
  id: "crm_relationship_statistics",
  label: new TranslatableMarkup("Relationship Statistics"),
  description: new TranslatableMarkup("Stores relationship type counts for a contact."),
  category: "crm",
  default_widget: NULL,
  default_formatter: "crm_relationship_statistics_default",
  no_ui: TRUE,
)]
class RelationshipStatisticsItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties = [];

    // The relationship type key (e.g., "friends" or "parent_child:a").
    // Named 'value' for list formatter compatibility.
    $properties['value'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Relationship type key'))
      ->setDescription(new TranslatableMarkup('The relationship type identifier.'))
      ->setRequired(TRUE);

    // The count of relationships of this type.
    $properties['count'] = DataDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Count'))
      ->setDescription(new TranslatableMarkup('The number of relationships of this type.'))
      ->setRequired(TRUE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    return [
      'columns' => [
        'value' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
        'count' => [
          'type' => 'int',
          'unsigned' => TRUE,
          'not null' => TRUE,
          'default' => 0,
        ],
      ],
      'indexes' => [
        'value' => ['value'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $value = $this->get('value')->getValue();
    return $value === NULL || $value === '';
  }

  /**
   * {@inheritdoc}
   */
  public static function mainPropertyName() {
    return 'value';
  }

}
