<?php

namespace Drupal\crm\Plugin\Menu\LocalTask;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Menu\LocalTaskDefault;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\crm\CrmContactInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides a local task that shows the amount of comments.
 */
class Comment extends LocalTaskDefault implements ContainerFactoryPluginInterface {
  use StringTranslationTrait;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Construct the Comment object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    RouteMatchInterface $route_match,
    ModuleHandlerInterface $module_handler,
  ) {

    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->routeMatch        = $route_match;
    $this->moduleHandler     = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('current_route_match'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getTitle(?Request $request = NULL) {
    // Return default title if comment module is not enabled.
    if (!$this->moduleHandler->moduleExists('comment')) {
      return $this->t('Comments');
    }

    $contact = $this->routeMatch->getParameter('crm_contact');
    // Check if contact has comment field.
    if ($contact instanceof CrmContactInterface && !$contact->hasField('comment')) {
      return $this->t('Comments');
    }

    $contact_id = $contact;
    if ($contact_id instanceof CrmContactInterface) {
      $contact_id = $contact_id->id();
    }

    try {
      $storage = $this->entityTypeManager->getStorage('comment');
      $count   = $storage->getQuery()
        ->condition('entity_type', 'crm_contact')
        ->condition('entity_id', $contact_id)
        ->condition('status', 1)
        ->condition('field_name', 'comment')
        ->accessCheck(TRUE)
        ->count()
        ->execute();

      return $this->formatPlural($count, 'Comment (1)', 'Comments (@count)');
    }
    catch (\Exception $e) {
      return $this->t('Comments');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {

    $contact_id = $this->routeMatch->getParameter('crm_contact');
    if ($contact_id instanceof CrmContactInterface) {
      $contact_id = $contact_id->id();
    }

    return ['crm_contact:' . $contact_id];
  }

}
