<?php

namespace Drupal\crm\Plugin\Validation\Constraint;

use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Validates that Contact A and Contact B are different.
 */
class RelationshipContactsConstraintValidator extends ConstraintValidator {

  /**
   * Validates the entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being validated.
   * @param \Drupal\crm\Plugin\Validation\Constraint\RelationshipContactsConstraint $constraint
   *   The constraint to validate against.
   */
  public function validate($entity, Constraint $constraint) {
    $contact_a = $entity->get('contacts')->referencedEntities()[0] ?? NULL;
    $contact_b = $entity->get('contacts')->referencedEntities()[1] ?? NULL;

    $contact_a_target_id = $contact_a ? $contact_a->id() : NULL;
    $contact_b_target_id = $contact_b ? $contact_b->id() : NULL;

    if ($contact_a_target_id === $contact_b_target_id) {
      $this->context->addViolation($constraint->differentMessage);
    }

    $relationship_type = $entity->get('bundle')->entity ?? NULL;
    $expected_contact_a_type = $relationship_type ? $relationship_type->get('contact_type_a') : [];
    $expected_contact_b_type = $relationship_type ? $relationship_type->get('contact_type_b') : [];

    $contact_a_type = $contact_a ? $contact_a->bundle() : NULL;
    $contact_b_type = $contact_b ? $contact_b->bundle() : NULL;

    if (!in_array($contact_a_type, $expected_contact_a_type)) {
      $this->context->buildViolation($constraint->wrongTypeMessage)
        ->atPath('contact_a')
        ->setParameter('@type', implode(', ', $expected_contact_a_type))
        ->addViolation();
    }

    if (!in_array($contact_b_type, $expected_contact_b_type)) {
      $this->context->buildViolation($constraint->wrongTypeMessage)
        ->atPath('contact_b')
        ->setParameter('@type', implode(', ', $expected_contact_b_type))
        ->addViolation();
    }

    // Validate valid contacts constraints.
    if ($relationship_type) {
      $valid_contacts_a = $relationship_type->getValidContactsA();
      $valid_contacts_b = $relationship_type->getValidContactsB();

      // Check Contact A is in valid contacts list (if configured).
      if (!empty($valid_contacts_a) && $contact_a_target_id !== NULL) {
        if (!in_array((int) $contact_a_target_id, array_map('intval', $valid_contacts_a))) {
          $this->context->buildViolation($constraint->invalidContactMessage)
            ->atPath('contact_a')
            ->addViolation();
        }
      }

      // Check Contact B is in valid contacts list (if configured).
      if (!empty($valid_contacts_b) && $contact_b_target_id !== NULL) {
        if (!in_array((int) $contact_b_target_id, array_map('intval', $valid_contacts_b))) {
          $this->context->buildViolation($constraint->invalidContactMessage)
            ->atPath('contact_b')
            ->addViolation();
        }
      }
    }
  }

}
