<?php

namespace Drupal\crm\Plugin\Validation\Constraint;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Validates that relationship limits are not exceeded.
 */
class RelationshipLimitConstraintValidator extends ConstraintValidator implements ContainerInjectionInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new RelationshipLimitConstraintValidator.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Validates the entity.
   *
   * @param \Drupal\crm\CrmRelationshipInterface $entity
   *   The entity being validated.
   * @param \Drupal\crm\Plugin\Validation\Constraint\RelationshipLimitConstraint $constraint
   *   The constraint to validate against.
   */
  public function validate($entity, Constraint $constraint) {
    $relationship_type = $entity->get('bundle')->entity;
    if (!$relationship_type) {
      return;
    }

    $limit_a = $relationship_type->getLimitA();
    $limit_b = $relationship_type->getLimitB();

    // If no limits are configured, nothing to validate.
    if ($limit_a === NULL && $limit_b === NULL) {
      return;
    }

    $contacts = $entity->get('contacts')->referencedEntities();
    $contact_a = $contacts[0] ?? NULL;
    $contact_b = $contacts[1] ?? NULL;

    $limit_active_only = $relationship_type->isLimitActiveOnly();

    // Check limit for Contact A.
    if ($limit_a !== NULL && $contact_a) {
      $count = $this->countExistingRelationships(
        $contact_a->id(),
        $relationship_type->id(),
        0,
        $limit_active_only,
        $entity->id()
      );

      if ($count >= $limit_a) {
        $this->context->buildViolation($constraint->limitExceededMessageA)
          ->atPath('contact_a')
          ->setParameter('@count', $count)
          ->setParameter('@type', $relationship_type->label())
          ->setParameter('@label', $relationship_type->get('label_a') ?: $relationship_type->label())
          ->setParameter('@limit', $limit_a)
          ->addViolation();
      }
    }

    // Check limit for Contact B.
    if ($limit_b !== NULL && $contact_b) {
      $count = $this->countExistingRelationships(
        $contact_b->id(),
        $relationship_type->id(),
        1,
        $limit_active_only,
        $entity->id()
      );

      if ($count >= $limit_b) {
        $this->context->buildViolation($constraint->limitExceededMessageB)
          ->atPath('contact_b')
          ->setParameter('@count', $count)
          ->setParameter('@type', $relationship_type->label())
          ->setParameter('@label', $relationship_type->get('label_b') ?: $relationship_type->label())
          ->setParameter('@limit', $limit_b)
          ->addViolation();
      }
    }
  }

  /**
   * Counts existing relationships for a contact in a specific position.
   *
   * @param int|string $contact_id
   *   The contact ID.
   * @param string $relationship_type_id
   *   The relationship type ID.
   * @param int $delta
   *   The position (0 for contact_a, 1 for contact_b).
   * @param bool $active_only
   *   Whether to count only active relationships.
   * @param int|string|null $exclude_id
   *   The relationship ID to exclude (for edits).
   *
   * @return int
   *   The count of existing relationships.
   */
  protected function countExistingRelationships(
    int|string $contact_id,
    string $relationship_type_id,
    int $delta,
    bool $active_only,
    int|string|null $exclude_id = NULL,
  ): int {
    $query = $this->entityTypeManager
      ->getStorage('crm_relationship')
      ->getQuery()
      ->condition('bundle', $relationship_type_id)
      ->condition('contacts', $contact_id)
      ->accessCheck(FALSE);

    if ($active_only) {
      $query->condition('status', 1);
    }

    if ($exclude_id !== NULL) {
      $query->condition('id', $exclude_id, '<>');
    }

    $relationship_ids = $query->execute();

    if (empty($relationship_ids)) {
      return 0;
    }

    // Load relationships and count those where the contact is at the
    // specified delta position.
    $relationships = $this->entityTypeManager
      ->getStorage('crm_relationship')
      ->loadMultiple($relationship_ids);

    $count = 0;
    foreach ($relationships as $relationship) {
      $contacts = $relationship->get('contacts')->getValue();
      if (isset($contacts[$delta]) && (string) $contacts[$delta]['target_id'] === (string) $contact_id) {
        $count++;
      }
    }

    return $count;
  }

}
