<?php

declare(strict_types=1);

namespace Drupal\crm\Plugin\views\field;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\views\Attribute\ViewsField;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Field handler for relationship statistics count.
 */
#[ViewsField("crm_relationship_statistics")]
class RelationshipStatistics extends FieldPluginBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Cached relationship types.
   *
   * @var array|null
   */
  protected ?array $relationshipTypes = NULL;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['format'] = ['default' => 'label_count'];
    $options['label_position'] = ['default' => 'opposite'];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $form['format'] = [
      '#type' => 'select',
      '#title' => $this->t('Display format'),
      '#options' => [
        'label_count' => $this->t('Label (count) - e.g., "Parent (3)"'),
        'label_colon_count' => $this->t('Label: count - e.g., "Parent: 3"'),
        'count_label' => $this->t('count Label - e.g., "3 Parents"'),
        'label_only' => $this->t('Label only'),
        'count_only' => $this->t('Count only'),
      ],
      '#default_value' => $this->options['format'],
    ];

    $form['label_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Label position'),
      '#description' => $this->t('For asymmetric relationships, choose which label to display.'),
      '#options' => [
        'opposite' => $this->t('Opposite position (show what the related contacts are)'),
        'same' => $this->t('Same position (show what this contact is in the relationship)'),
      ],
      '#default_value' => $this->options['label_position'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $value = $this->getValue($values);
    $count = $values->{$this->tableAlias . '_relationship_statistics_count'} ?? $values->relationship_statistics_count ?? 0;
    $type_key = $values->{$this->tableAlias . '_relationship_statistics_value'} ?? $values->relationship_statistics_value ?? '';

    if ($value === NULL && $count === 0) {
      return '';
    }

    // Use count from the field value if available.
    if (is_numeric($value)) {
      $count = (int) $value;
    }

    $format = $this->options['format'];
    $label_position = $this->options['label_position'] ?? 'opposite';
    $use_opposite = $label_position === 'opposite';

    // For 'label_only' format, always use singular label since count is not
    // displayed. For other formats, use singular/plural based on count.
    $label_count = $format === 'label_only' ? 1 : $count;
    $label = $this->getLabel($type_key, $use_opposite, $label_count);

    return match ($format) {
      'label_count' => new TranslatableMarkup('@label (@count)', ['@label' => $label, '@count' => $count]),
      'label_colon_count' => new TranslatableMarkup('@label: @count', ['@label' => $label, '@count' => $count]),
      'count_label' => new TranslatableMarkup('@count @label', ['@count' => $count, '@label' => $label]),
      'label_only' => $label,
      'count_only' => (string) $count,
      default => new TranslatableMarkup('@label (@count)', ['@label' => $label, '@count' => $count]),
    };
  }

  /**
   * Gets the human-readable label for a relationship type key.
   *
   * @param string $type_key
   *   The relationship type key (e.g., "friends" or "parent_child:a").
   * @param bool $use_opposite
   *   If TRUE, return the opposite position's label (what the related contacts
   *   are). If FALSE, return the same position's label (what this contact is
   *   in the relationship). Defaults to TRUE.
   * @param int $count
   *   The count of relationships. Used to determine singular vs plural label.
   *   Defaults to 1 (singular).
   *
   * @return string
   *   The human-readable label.
   */
  protected function getLabel(string $type_key, bool $use_opposite = TRUE, int $count = 1): string {
    if (empty($type_key)) {
      return '';
    }

    // Parse the type key to get the relationship type ID and position.
    $parts = explode(':', $type_key);
    $type_id = $parts[0];
    $position = $parts[1] ?? NULL;

    $relationship_types = $this->getRelationshipTypes();

    if (!isset($relationship_types[$type_id])) {
      // Fallback to the raw key if type not found.
      return $type_key;
    }

    $type = $relationship_types[$type_id];

    // Determine if we should use plural (count != 1).
    $use_plural = $count !== 1;

    // For asymmetric relationships, get the appropriate label based on
    // the use_opposite setting.
    if ($position === 'a') {
      // Position A: opposite means label_b, same means label_a.
      $label_key = $use_opposite ? 'label_b' : 'label_a';
      $plural_key = $use_opposite ? 'label_b_plural' : 'label_a_plural';
      $singular_label = $type->get($label_key) ?? $type->label();
      if ($use_plural) {
        $plural_label = $type->get($plural_key);
        return !empty($plural_label) ? $plural_label : $singular_label;
      }
      return $singular_label;
    }
    elseif ($position === 'b') {
      // Position B: opposite means label_a, same means label_b.
      $label_key = $use_opposite ? 'label_a' : 'label_b';
      $plural_key = $use_opposite ? 'label_a_plural' : 'label_b_plural';
      $singular_label = $type->get($label_key) ?? $type->label();
      if ($use_plural) {
        $plural_label = $type->get($plural_key);
        return !empty($plural_label) ? $plural_label : $singular_label;
      }
      return $singular_label;
    }

    // For symmetric relationships, use the main label.
    return $type->label();
  }

  /**
   * Gets all relationship types, cached for the request.
   *
   * @return array
   *   An array of relationship type entities keyed by ID.
   */
  protected function getRelationshipTypes(): array {
    if ($this->relationshipTypes === NULL) {
      $this->relationshipTypes = $this->entityTypeManager
        ->getStorage('crm_relationship_type')
        ->loadMultiple();
    }
    return $this->relationshipTypes;
  }

}
