<?php

declare(strict_types=1);

namespace Drupal\crm\Plugin\views\filter;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\views\Attribute\ViewsFilter;
use Drupal\views\Plugin\views\filter\InOperator;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filter handler for relationship statistics type.
 */
#[ViewsFilter("crm_relationship_statistics_type")]
class RelationshipStatisticsType extends InOperator {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getValueOptions() {
    if (!isset($this->valueOptions)) {
      $this->valueOptions = $this->getRelationshipTypeOptions();
    }
    return $this->valueOptions;
  }

  /**
   * Gets the relationship type options for the filter.
   *
   * @return array
   *   An array of options keyed by type key.
   */
  protected function getRelationshipTypeOptions(): array {
    $options = [];

    try {
      $types = $this->entityTypeManager
        ->getStorage('crm_relationship_type')
        ->loadMultiple();

      foreach ($types as $type) {
        if ($type->get('asymmetric')) {
          // For asymmetric relationships, add both positions.
          $label_a = $type->get('label_a') ?? $type->label();
          $label_b = $type->get('label_b') ?? $type->label();
          $options[$type->id() . ':a'] = new TranslatableMarkup('@type: @label', [
            '@type' => $type->label(),
            '@label' => $label_a,
          ]);
          $options[$type->id() . ':b'] = new TranslatableMarkup('@type: @label', [
            '@type' => $type->label(),
            '@label' => $label_b,
          ]);
        }
        else {
          // For symmetric relationships, use the main label.
          $options[$type->id()] = $type->label();
        }
      }
    }
    catch (\Exception $e) {
      // Return empty options if entity type manager fails.
    }

    return $options;
  }

}
