<?php

declare(strict_types=1);

namespace Drupal\crm\Plugin\views\sort;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\views\Attribute\ViewsSort;
use Drupal\views\Plugin\views\sort\SortPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Sort handler for relationship statistics count.
 */
#[ViewsSort("crm_relationship_statistics_count")]
class RelationshipStatisticsCount extends SortPluginBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['relationship_type'] = ['default' => ''];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $form['relationship_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Relationship type'),
      '#description' => $this->t('Select the relationship type to sort by. Leave empty to sort by any relationship count.'),
      '#options' => ['' => $this->t('- Any -')] + $this->getRelationshipTypeOptions(),
      '#default_value' => $this->options['relationship_type'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function query(): void {
    $this->ensureMyTable();

    $type_key = $this->options['relationship_type'] ?? '';

    // If a specific relationship type is selected, add a condition.
    if ($type_key) {
      $this->query->addWhere(0, "$this->tableAlias.relationship_statistics_value", $type_key);
    }

    // Add the order by clause.
    $this->query->addOrderBy($this->tableAlias, 'relationship_statistics_count', $this->options['order']);
  }

  /**
   * Gets the relationship type options for the settings form.
   *
   * @return array
   *   An array of options keyed by type key.
   */
  protected function getRelationshipTypeOptions(): array {
    $options = [];

    try {
      $types = $this->entityTypeManager
        ->getStorage('crm_relationship_type')
        ->loadMultiple();

      foreach ($types as $type) {
        if ($type->get('asymmetric')) {
          // For asymmetric relationships, add both positions.
          $label_a = $type->get('label_a') ?? $type->label();
          $label_b = $type->get('label_b') ?? $type->label();
          $options[$type->id() . ':a'] = new TranslatableMarkup('@type: @label', [
            '@type' => $type->label(),
            '@label' => $label_a,
          ]);
          $options[$type->id() . ':b'] = new TranslatableMarkup('@type: @label', [
            '@type' => $type->label(),
            '@label' => $label_b,
          ]);
        }
        else {
          // For symmetric relationships, use the main label.
          $options[$type->id()] = $type->label();
        }
      }
    }
    catch (\Exception $e) {
      // Return empty options if entity type manager fails.
    }

    return $options;
  }

}
