<?php

namespace Drupal\crm;

use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Service description.
 */
class Relationship {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;


  /**
   * The relationship storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $relationshipStorage;

  /**
   * Constructs a Relationship object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->relationshipStorage = $entity_type_manager
      ->getStorage('crm_relationship');
  }

  /**
   * Method description.
   */
  public function getRelationshipIdByContactId(int $contact_id) {
    $query = $this->relationshipStorage->getQuery();
    $ids = $query
      ->condition('status', 1)
      ->condition('contacts', $contact_id)
      ->accessCheck(TRUE)
      ->execute();

    return $ids;
  }

  /**
   * Gets eligible relationship types for a contact.
   *
   * Filters relationship types based on the contact's bundle matching
   * the allowed contact types for each position (A or B), and optionally
   * the valid_contacts restrictions if configured.
   *
   * @param \Drupal\crm\CrmContactInterface $contact
   *   The contact entity to check eligibility for.
   *
   * @return array
   *   An array of eligible relationship types, keyed by type ID, containing:
   *   - type: The relationship type entity.
   *   - positions: Array of positions ('a', 'b') the contact can occupy.
   */
  public function getEligibleRelationshipTypesForContact(CrmContactInterface $contact): array {
    $contact_type = $contact->bundle();
    $contact_id = (int) $contact->id();
    $eligible = [];

    $relationship_types = $this->entityTypeManager
      ->getStorage('crm_relationship_type')
      ->loadMultiple();

    foreach ($relationship_types as $type) {
      $positions = [];

      // Check position A eligibility.
      $contact_type_a = $type->get('contact_type_a') ?? [];
      if (!is_array($contact_type_a)) {
        $contact_type_a = $contact_type_a !== NULL && $contact_type_a !== '' ? [$contact_type_a] : [];
      }
      if (in_array($contact_type, $contact_type_a, TRUE)) {
        // Check valid_contacts_a restriction if configured.
        $valid_contacts_a = $type->getValidContactsA();
        if (empty($valid_contacts_a) || in_array($contact_id, array_map('intval', $valid_contacts_a), TRUE)) {
          $positions[] = 'a';
        }
      }

      // Check position B eligibility.
      $contact_type_b = $type->get('contact_type_b') ?? [];
      if (!is_array($contact_type_b)) {
        $contact_type_b = $contact_type_b !== NULL && $contact_type_b !== '' ? [$contact_type_b] : [];
      }
      if (in_array($contact_type, $contact_type_b, TRUE)) {
        // Check valid_contacts_b restriction if configured.
        $valid_contacts_b = $type->getValidContactsB();
        if (empty($valid_contacts_b) || in_array($contact_id, array_map('intval', $valid_contacts_b), TRUE)) {
          $positions[] = 'b';
        }
      }

      if (!empty($positions)) {
        $eligible[$type->id()] = [
          'type' => $type,
          'positions' => $positions,
        ];
      }
    }

    return $eligible;
  }

}
